# Email Setup Guide - Send Real Emails

## Overview
This guide shows you how to configure Django to send real emails when leads are converted to projects (and for other notifications).

## Option 1: Gmail SMTP (Easiest for Testing)

### Step 1: Enable Gmail App Password

1. Go to your Google Account: https://myaccount.google.com/
2. Click "Security" in the left sidebar
3. Enable "2-Step Verification" (if not already enabled)
4. Search for "App passwords" or go to: https://myaccount.google.com/apppasswords
5. Select "Mail" and "Windows Computer" (or Other)
6. Click "Generate"
7. Copy the 16-character password (example: `abcd efgh ijkl mnop`)

### Step 2: Configure Django Settings

Add to `myproject/settings.py`:

```python
# Email Configuration
EMAIL_BACKEND = 'django.core.mail.backends.smtp.EmailBackend'
EMAIL_HOST = 'smtp.gmail.com'
EMAIL_PORT = 587
EMAIL_USE_TLS = True
EMAIL_HOST_USER = 'your-email@gmail.com'  # Your Gmail address
EMAIL_HOST_PASSWORD = 'abcdefghijklmnop'  # Your 16-char app password (no spaces)
DEFAULT_FROM_EMAIL = 'Immigration Services <your-email@gmail.com>'
SERVER_EMAIL = 'your-email@gmail.com'
```

### Step 3: Test Email Configuration

Create `test_email.py`:

```python
from django.core.mail import send_mail
from django.conf import settings

# Test email
send_mail(
    subject='Test Email from Django',
    message='This is a test email to verify email configuration.',
    from_email=settings.DEFAULT_FROM_EMAIL,
    recipient_list=['recipient@example.com'],  # Change to your email
    fail_silently=False,
)

print("Email sent successfully!")
```

Run:
```bash
python manage.py shell < test_email.py
```

## Option 2: SendGrid (Recommended for Production)

### Step 1: Create SendGrid Account

1. Sign up at: https://sendgrid.com/
2. Free tier: 100 emails/day
3. Verify your email address
4. Create an API key:
   - Go to Settings → API Keys
   - Click "Create API Key"
   - Name it "Django App"
   - Select "Full Access"
   - Copy the API key (starts with `SG.`)

### Step 2: Install SendGrid Package

```bash
pip install sendgrid
```

### Step 3: Configure Django Settings

Add to `myproject/settings.py`:

```python
# Email Configuration - SendGrid
EMAIL_BACKEND = 'django.core.mail.backends.smtp.EmailBackend'
EMAIL_HOST = 'smtp.sendgrid.net'
EMAIL_PORT = 587
EMAIL_USE_TLS = True
EMAIL_HOST_USER = 'apikey'  # This is literally the string 'apikey'
EMAIL_HOST_PASSWORD = 'SG.your-api-key-here'  # Your SendGrid API key
DEFAULT_FROM_EMAIL = 'Immigration Services <noreply@yourdomain.com>'
SERVER_EMAIL = 'noreply@yourdomain.com'
```

## Option 3: AWS SES (Best for Production)

### Step 1: Setup AWS SES

1. Go to AWS Console: https://console.aws.amazon.com/ses/
2. Verify your domain or email address
3. Request production access (initially in sandbox mode)
4. Create SMTP credentials:
   - Go to "SMTP Settings"
   - Click "Create My SMTP Credentials"
   - Download credentials

### Step 2: Configure Django Settings

Add to `myproject/settings.py`:

```python
# Email Configuration - AWS SES
EMAIL_BACKEND = 'django.core.mail.backends.smtp.EmailBackend'
EMAIL_HOST = 'email-smtp.us-east-1.amazonaws.com'  # Change region if needed
EMAIL_PORT = 587
EMAIL_USE_TLS = True
EMAIL_HOST_USER = 'your-smtp-username'  # From AWS SES SMTP credentials
EMAIL_HOST_PASSWORD = 'your-smtp-password'  # From AWS SES SMTP credentials
DEFAULT_FROM_EMAIL = 'Immigration Services <noreply@yourdomain.com>'
SERVER_EMAIL = 'noreply@yourdomain.com'
```

## Option 4: Development/Testing (Console Backend)

For testing without sending real emails:

```python
# Email Configuration - Console (for development)
EMAIL_BACKEND = 'django.core.mail.backends.console.EmailBackend'
```

Emails will be printed to console instead of sent.

## Implementation: Update Lead Conversion

I'll update the lead conversion function to send emails:

### File: `crm/views.py`

Find the `admin_lead_convert` function and update the email section:

```python
# After creating client account, send email
if not client_user:  # New account created
    from django.core.mail import send_mail
    from django.conf import settings
    
    try:
        # Send welcome email with credentials
        email_subject = 'Welcome to Immigration Services - Your Account Details'
        email_message = f'''
Dear {client_user.first_name} {client_user.last_name},

Welcome to Immigration Services!

Your account has been created and you can now access our client portal.

LOGIN CREDENTIALS:
------------------
Username: {username}
Password: {password}

Login URL: http://127.0.0.1:8000/client/login/
(Replace with your actual domain in production)

IMPORTANT SECURITY NOTICE:
- Please change your password after your first login
- Keep your credentials secure
- Do not share your password with anyone

Your project "{project.title}" has been created and is ready for you to review.

If you have any questions, please don't hesitate to contact us.

Best regards,
Immigration Services Team
        '''
        
        send_mail(
            subject=email_subject,
            message=email_message,
            from_email=settings.DEFAULT_FROM_EMAIL,
            recipient_list=[client_user.email],
            fail_silently=False,  # Set to True in production to prevent errors
        )
        
        print(f"Welcome email sent to {client_user.email}")
        
    except Exception as e:
        print(f"Error sending email: {str(e)}")
        # Continue even if email fails
```

## HTML Email Template (Optional but Recommended)

Create `crm/templates/emails/welcome_email.html`:

```html
<!DOCTYPE html>
<html>
<head>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
        .header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0; }
        .content { background: #f9f9f9; padding: 30px; border-radius: 0 0 10px 10px; }
        .credentials { background: white; padding: 20px; border-left: 4px solid #667eea; margin: 20px 0; }
        .button { display: inline-block; padding: 12px 30px; background: #667eea; color: white; text-decoration: none; border-radius: 5px; margin: 20px 0; }
        .footer { text-align: center; color: #666; font-size: 12px; margin-top: 30px; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Welcome to Immigration Services!</h1>
        </div>
        <div class="content">
            <p>Dear {{ client_name }},</p>
            
            <p>Your account has been created and you can now access our client portal.</p>
            
            <div class="credentials">
                <h3>Your Login Credentials:</h3>
                <p><strong>Username:</strong> {{ username }}</p>
                <p><strong>Password:</strong> {{ password }}</p>
            </div>
            
            <a href="{{ login_url }}" class="button">Login to Your Account</a>
            
            <p><strong>Important Security Notice:</strong></p>
            <ul>
                <li>Please change your password after your first login</li>
                <li>Keep your credentials secure</li>
                <li>Do not share your password with anyone</li>
            </ul>
            
            <p>Your project "<strong>{{ project_title }}</strong>" has been created and is ready for you to review.</p>
            
            <p>If you have any questions, please don't hesitate to contact us.</p>
            
            <p>Best regards,<br>Immigration Services Team</p>
        </div>
        <div class="footer">
            <p>&copy; 2026 Immigration Services. All rights reserved.</p>
        </div>
    </div>
</body>
</html>
```

### Send HTML Email

```python
from django.core.mail import EmailMultiAlternatives
from django.template.loader import render_to_string

# Render HTML template
html_content = render_to_string('emails/welcome_email.html', {
    'client_name': f"{client_user.first_name} {client_user.last_name}",
    'username': username,
    'password': password,
    'login_url': 'http://127.0.0.1:8000/client/login/',
    'project_title': project.title,
})

# Create email
email = EmailMultiAlternatives(
    subject='Welcome to Immigration Services - Your Account Details',
    body=email_message,  # Plain text version
    from_email=settings.DEFAULT_FROM_EMAIL,
    to=[client_user.email]
)

# Attach HTML version
email.attach_alternative(html_content, "text/html")

# Send
email.send(fail_silently=False)
```

## Environment Variables (Recommended for Security)

Instead of hardcoding credentials in settings.py, use environment variables:

### Step 1: Install python-decouple

```bash
pip install python-decouple
```

### Step 2: Create `.env` file

Create `.env` in project root:

```env
# Email Configuration
EMAIL_HOST_USER=your-email@gmail.com
EMAIL_HOST_PASSWORD=your-app-password
DEFAULT_FROM_EMAIL=Immigration Services <your-email@gmail.com>
```

### Step 3: Update settings.py

```python
from decouple import config

# Email Configuration
EMAIL_BACKEND = 'django.core.mail.backends.smtp.EmailBackend'
EMAIL_HOST = 'smtp.gmail.com'
EMAIL_PORT = 587
EMAIL_USE_TLS = True
EMAIL_HOST_USER = config('EMAIL_HOST_USER')
EMAIL_HOST_PASSWORD = config('EMAIL_HOST_PASSWORD')
DEFAULT_FROM_EMAIL = config('DEFAULT_FROM_EMAIL')
SERVER_EMAIL = config('EMAIL_HOST_USER')
```

### Step 4: Add .env to .gitignore

```
.env
*.pyc
__pycache__/
db.sqlite3
```

## Testing Checklist

- [ ] Email settings configured in settings.py
- [ ] Test email sent successfully
- [ ] Lead conversion sends email
- [ ] Email contains correct credentials
- [ ] Email formatting looks good
- [ ] Links in email work correctly
- [ ] Email arrives in inbox (not spam)
- [ ] Client can login with emailed credentials

## Troubleshooting

### Email not sending

1. **Check settings**: Verify EMAIL_HOST_USER and EMAIL_HOST_PASSWORD
2. **Check firewall**: Port 587 must be open
3. **Check spam folder**: Email might be in spam
4. **Enable less secure apps**: For Gmail (not recommended)
5. **Check logs**: Look for error messages in console

### Gmail "Less secure app" error

- Use App Password instead of regular password
- Enable 2-Step Verification first
- Generate App Password from Google Account settings

### SendGrid not working

- Verify API key is correct
- Check SendGrid dashboard for errors
- Verify sender email is verified in SendGrid

### Emails going to spam

- Use verified domain
- Add SPF and DKIM records
- Use professional email service (SendGrid, AWS SES)
- Avoid spam trigger words in subject/body

## Production Recommendations

1. **Use Professional Email Service**: SendGrid, AWS SES, Mailgun
2. **Verify Domain**: Add SPF, DKIM, DMARC records
3. **Use Environment Variables**: Never commit credentials
4. **Enable SSL/TLS**: Always use encrypted connection
5. **Handle Failures Gracefully**: Use `fail_silently=True` in production
6. **Log Email Sends**: Track all emails sent
7. **Rate Limiting**: Respect email service limits
8. **Unsubscribe Link**: Add for marketing emails
9. **Email Templates**: Use professional HTML templates
10. **Test Thoroughly**: Test on multiple email clients

## Quick Start (Gmail)

1. Get Gmail App Password
2. Add to `settings.py`:
```python
EMAIL_BACKEND = 'django.core.mail.backends.smtp.EmailBackend'
EMAIL_HOST = 'smtp.gmail.com'
EMAIL_PORT = 587
EMAIL_USE_TLS = True
EMAIL_HOST_USER = 'your-email@gmail.com'
EMAIL_HOST_PASSWORD = 'your-16-char-app-password'
DEFAULT_FROM_EMAIL = 'Immigration Services <your-email@gmail.com>'
```
3. Test with `test_email.py`
4. Done!

## Summary

To send real emails when converting leads:
1. Choose email service (Gmail for testing, SendGrid/AWS SES for production)
2. Configure Django settings with SMTP credentials
3. Update lead conversion function to send email
4. Test thoroughly
5. Monitor email delivery

Your clients will now receive professional welcome emails with their login credentials!
