# Lead to Project & Staff Registration System Changes

## Overview
This document outlines the changes needed for:
1. Lead conversion without creating user accounts
2. Staff self-registration with admin approval
3. Automatic project linking when clients register

## Part 1: Lead Conversion Without User Creation

### Current Issue
- When leads are converted to projects, user accounts are created automatically
- This prevents the actual person from registering with their email later

### Solution
**Don't create user accounts during lead conversion. Instead:**
- Store the lead's email with the project
- When a user registers with that email, automatically link existing projects to their account

### Changes Required:

#### 1. Update Project Model
Add field to store client email even without user account:
```python
class Project(models.Model):
    client = models.ForeignKey(User, null=True, blank=True)  # Make nullable
    client_email = models.EmailField(null=True, blank=True)  # Store email
    # ... rest of fields
```

#### 2. Update Lead Conversion (admin_lead_convert)
Remove user creation logic:
```python
# OLD: Creates user account
# NEW: Just store email and create project
project = Project.objects.create(
    lead=lead,
    service=service,
    client=None,  # No user yet
    client_email=lead.email,  # Store email
    title=f"{service.name} for {lead.get_full_name()}",
    description=service.description,
    budget=service.price,
    status='pending_payment'
)
```

#### 3. Update Client Registration (client_register)
Check for existing projects with matching email:
```python
# After creating user account
user = User.objects.create_user(...)

# Link any existing projects
projects = Project.objects.filter(client_email=email, client__isnull=True)
for project in projects:
    project.client = user
    project.save()
    # Notify user about existing projects
```

## Part 2: Staff Self-Registration with Admin Approval

### Requirements
- Staff can register themselves
- They cannot login until admin approves
- Only applies to staff roles (CRM, PM, Employee, Lead Manager)
- Admin accounts are created by superuser only

### Solution
Add approval system using UserProfile model

### Changes Required:

#### 1. Update UserProfile Model
Add approval fields:
```python
class UserProfile(models.Model):
    # ... existing fields
    is_approved = models.BooleanField(default=False)
    approval_requested_at = models.DateTimeField(null=True, blank=True)
    approved_by = models.ForeignKey(User, null=True, blank=True, related_name='approved_staff')
    approved_at = models.DateTimeField(null=True, blank=True)
```

#### 2. Create Staff Registration View
New view for staff to register:
```python
def staff_register(request):
    # Allow staff to choose role (CRM, PM, Employee, Lead Manager)
    # Create user with is_active=False
    # Create profile with is_approved=False
    # Notify admin of pending approval
```

#### 3. Update Staff Login
Check approval status:
```python
def staff_login(request):
    user = authenticate(...)
    if user:
        if user.profile.role != 'client':
            if not user.profile.is_approved:
                return error: "Your account is pending admin approval"
        login(request, user)
```

#### 4. Create Admin Approval Interface
Add to admin dashboard:
- List of pending staff registrations
- Approve/Reject buttons
- When approved: set is_approved=True, user.is_active=True

## Implementation Priority

### Phase 1: Lead Conversion Changes (High Priority)
1. Add migration for Project.client_email field
2. Update admin_lead_convert to not create users
3. Update client_register to link existing projects
4. Test lead conversion and client registration flow

### Phase 2: Staff Registration (Medium Priority)
1. Add migration for UserProfile approval fields
2. Create staff registration page and view
3. Update staff login to check approval
4. Create admin approval interface
5. Test staff registration and approval flow

## Database Migration Required

```python
# Migration for Project model
python manage.py makemigrations
python manage.py migrate

# This will:
# - Add client_email field to Project
# - Make client field nullable
# - Add approval fields to UserProfile
```

## Testing Checklist

### Lead Conversion Testing
- [ ] Convert lead to project without creating user
- [ ] Verify project has client_email but no client user
- [ ] Register new client with same email
- [ ] Verify projects are automatically linked
- [ ] Check notifications are sent

### Staff Registration Testing
- [ ] Staff registers with role selection
- [ ] Verify cannot login before approval
- [ ] Admin sees pending registration
- [ ] Admin approves registration
- [ ] Staff can now login successfully
- [ ] Verify client registration still works normally

## Notes
- Existing projects with client accounts will continue to work
- This is a non-breaking change for existing data
- Email matching is case-insensitive
- Consider adding email verification for security

## Estimated Implementation Time
- Phase 1: 2-3 hours
- Phase 2: 3-4 hours
- Testing: 2 hours
- Total: 7-9 hours

Would you like me to proceed with implementing these changes?
