# Staff Role Management - Complete Implementation

## Overview
The CRM system now properly manages staff roles with clear separation between different user types.

## User Types in the System

### 1. Superusers (Admins)
- Created via Django's `createsuperuser` command
- Have full system access
- NOT shown in the CRM users list
- NOT available as a role option when creating users
- Access the admin dashboard at `/crm-admin/dashboard/`

### 2. Staff Roles (Created via CRM Dashboard)
These are the 4 staff roles available when creating users:

#### CRM Staff
- Role: `crm`
- Manages client relationships
- Handles projects assigned to them
- Dashboard: `/crm/dashboard/`

#### Lead Manager
- Role: `lead_manager`
- Manages and converts leads
- Tracks lead pipeline
- Dashboard: `/lead-manager/dashboard/`

#### Project Manager
- Role: `project_manager`
- Manages projects and tasks
- Oversees project delivery
- Dashboard: `/pm/dashboard/`

#### Employee
- Role: `employee`
- Completes assigned tasks
- Works on projects
- Dashboard: `/employee/dashboard/`

### 3. Clients
- Self-register at `/client/register/`
- NOT available as a role option when creating users
- NOT shown in the staff users list
- Access client dashboard at `/client/dashboard/`

## Admin Dashboard Statistics

The admin dashboard shows 11 stat cards with separate counts:

1. **Total Leads** - All leads in the system
2. **Converted Leads** - Leads converted to projects
3. **Total Projects** - All projects
4. **Pending Payments** - Payments awaiting approval
5. **Total Revenue** - Sum of approved payments
6. **Total Clients** - Count of registered clients
7. **Active Projects** - Projects currently in progress
8. **CRM Staff** - Count of users with CRM role
9. **Lead Managers** - Count of users with Lead Manager role
10. **Project Managers** - Count of users with Project Manager role
11. **Employees** - Count of users with Employee role

## User Creation Form

### Available Roles in Dropdown
When creating a new user via `/crm-admin/users/add/`, only these 4 staff roles are shown:
- CRM
- Lead Manager
- Project Manager
- Employee

### NOT Available in Dropdown
- Admin (created via command line only)
- Client (self-register only)

## Users List Page

### What's Shown
The users list at `/crm-admin/users/` shows:
- All staff members (CRM, Lead Manager, Project Manager, Employee)
- All clients (if any)

### What's NOT Shown
- Superusers/Admins
- Users with admin role

This keeps the users list focused on manageable staff and clients, not system administrators.

## Current System State

### Staff Members
```
Username    | Email                  | Role
------------|------------------------|------------------
crm         | crm@gmail.com          | CRM
lead        | lead@gmail.com         | Lead Manager
project     | project@gmail.com      | Project Manager
employee    | employee@gmail.com     | Employee
```

### Superusers (Not in Users List)
```
Username    | Email                  | Role
------------|------------------------|------------------
admin       | admin@gmail.com        | Superuser
```

### Clients
```
Currently: 0 registered clients
```

## How to Create Different User Types

### Create Superuser (Admin)
```bash
python manage.py createsuperuser
```

### Create Staff Member
1. Login as admin at `/staff/login/`
2. Go to `/crm-admin/users/add/`
3. Fill in the form
4. Select role: CRM, Lead Manager, Project Manager, or Employee
5. Click "Create User"

### Create Client
Clients self-register at `/client/register/`

## Access URLs

### Admin
- Login: `/staff/login/`
- Dashboard: `/crm-admin/dashboard/`
- Leads: `/crm-admin/leads/`
- Projects: `/crm-admin/projects/`
- Payments: `/crm-admin/payments/`
- Services: `/crm-admin/services/`
- Users: `/crm-admin/users/`
- Add User: `/crm-admin/users/add/`

### CRM Staff
- Login: `/staff/login/`
- Dashboard: `/crm/dashboard/`

### Lead Manager
- Login: `/staff/login/`
- Dashboard: `/lead-manager/dashboard/`

### Project Manager
- Login: `/staff/login/`
- Dashboard: `/pm/dashboard/`

### Employee
- Login: `/staff/login/`
- Dashboard: `/employee/dashboard/`

### Client
- Login: `/client/login/`
- Register: `/client/register/`
- Dashboard: `/client/dashboard/`

## Implementation Details

### Views Update
File: `crm/views.py`

```python
@login_required
def admin_users(request):
    """Admin users management page"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    # Get all users with profiles, excluding superusers/admins
    users = User.objects.filter(is_superuser=False).exclude(profile__role='admin').order_by('-date_joined')
    
    # Apply filters
    role_filter = request.GET.get('role')
    
    if role_filter:
        users = users.filter(profile__role=role_filter)
    
    context = {
        'user': user,
        'users': users
    }
    
    return render(request, 'crm/admin/users.html', context)
```

### Dashboard Stats
File: `crm/views.py`

```python
stats = {
    'total_leads': Lead.objects.count(),
    'new_leads': Lead.objects.filter(status='new').count(),
    'converted_leads': Lead.objects.filter(status='converted').count(),
    'total_projects': Project.objects.count(),
    'active_projects': Project.objects.filter(status='active').count(),
    'pending_payments': Payment.objects.filter(status='pending').count(),
    'total_revenue': Payment.objects.filter(status='approved').aggregate(Sum('amount'))['amount__sum'] or 0,
    'total_clients': User.objects.filter(profile__role='client').count(),
    'crm_count': User.objects.filter(profile__role='crm').count(),
    'lead_manager_count': User.objects.filter(profile__role='lead_manager').count(),
    'project_manager_count': User.objects.filter(profile__role='project_manager').count(),
    'employee_count': User.objects.filter(profile__role='employee').count(),
}
```

### User Creation Form
File: `crm/templates/crm/admin/user_add.html`

```html
<select name="role" required>
    <option value="">Select Role</option>
    <option value="crm">CRM</option>
    <option value="lead_manager">Lead Manager</option>
    <option value="project_manager">Project Manager</option>
    <option value="employee">Employee</option>
</select>
<div class="help-text">Staff roles only. Admins are superusers, clients self-register.</div>
```

## Verification

Run the verification script to check counts:
```bash
python check_staff_counts.py
```

This will show:
- Count of each staff role
- List of all users (excluding superusers)
- List of superusers (not shown in users list)

## Summary

✓ Admin and Client roles removed from user creation dropdown
✓ Dashboard shows separate counts for each staff role (CRM, Lead Manager, Project Manager, Employee)
✓ Users list excludes superusers and admin role users
✓ Clear separation between system admins, staff, and clients
✓ All role-based access control working correctly

The system now properly manages different user types with appropriate visibility and access controls.
