from django.db import models
from django.core.validators import MinValueValidator, MaxValueValidator
from cryptography.fernet import Fernet
from django.conf import settings
import base64
import hashlib
import uuid

class ProfileAssessment(models.Model):
    """Model for storing EB-1A profile assessments with encryption"""
    
    # Unique identifier
    assessment_id = models.UUIDField(default=uuid.uuid4, editable=False, unique=True)
    
    # Encrypted Personal Information
    first_name_encrypted = models.BinaryField(default=b'')
    last_name_encrypted = models.BinaryField(default=b'')
    email_encrypted = models.BinaryField(default=b'')
    phone_number_encrypted = models.BinaryField(null=True, blank=True, default=b'')
    country_code = models.CharField(max_length=5, default='+1')
    current_location_encrypted = models.BinaryField(default=b'')
    field_of_expertise = models.CharField(max_length=200)
    years_of_experience = models.CharField(max_length=50)
    
    # EB-1A Criteria Scores (0-3 scale)
    original_contributions = models.IntegerField(
        validators=[MinValueValidator(0), MaxValueValidator(3)],
        default=0,
        help_text="Original scientific, scholarly, artistic contributions"
    )
    published_material = models.IntegerField(
        validators=[MinValueValidator(0), MaxValueValidator(3)],
        default=0,
        help_text="Published material about you"
    )
    judging_work = models.IntegerField(
        validators=[MinValueValidator(0), MaxValueValidator(3)],
        default=0,
        help_text="Judging others' work"
    )
    scholarly_articles = models.IntegerField(
        validators=[MinValueValidator(0), MaxValueValidator(3)],
        default=0,
        help_text="Authored scholarly articles"
    )
    leading_role = models.IntegerField(
        validators=[MinValueValidator(0), MaxValueValidator(3)],
        default=0,
        help_text="Leading role in organizations"
    )
    high_salary = models.IntegerField(
        validators=[MinValueValidator(0), MaxValueValidator(3)],
        default=0,
        help_text="High salary/remuneration"
    )
    exhibitions = models.IntegerField(
        validators=[MinValueValidator(0), MaxValueValidator(3)],
        default=0,
        help_text="Exhibitions/showcases"
    )
    membership = models.IntegerField(
        validators=[MinValueValidator(0), MaxValueValidator(3)],
        default=0,
        help_text="Membership in associations"
    )
    awards = models.IntegerField(
        validators=[MinValueValidator(0), MaxValueValidator(3)],
        default=0,
        help_text="Awards & recognition"
    )
    commercial_success = models.IntegerField(
        validators=[MinValueValidator(0), MaxValueValidator(3)],
        default=0,
        help_text="Commercial success"
    )
    
    # Calculated fields
    total_score = models.IntegerField(default=0)
    percentage_score = models.FloatField(default=0.0)
    criteria_met = models.IntegerField(default=0)
    eligibility_status = models.CharField(max_length=50, default='Not Eligible')
    
    # Timestamps
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    
    class Meta:
        ordering = ['-created_at']
        verbose_name = 'Profile Assessment'
        verbose_name_plural = 'Profile Assessments'
    
    def __str__(self):
        return f"{self.get_first_name()} {self.get_last_name()} - {self.get_email()}"
    
    @staticmethod
    def get_cipher():
        """Get encryption cipher"""
        key = hashlib.sha256(settings.SECRET_KEY.encode()).digest()
        key_base64 = base64.urlsafe_b64encode(key)
        return Fernet(key_base64)
    
    def encrypt_field(self, value):
        """Encrypt a field value"""
        if not value:
            return b''
        cipher = self.get_cipher()
        return cipher.encrypt(value.encode())
    
    def decrypt_field(self, encrypted_value):
        """Decrypt a field value"""
        if not encrypted_value:
            return ''
        cipher = self.get_cipher()
        return cipher.decrypt(encrypted_value).decode()
    
    # Setter methods
    def set_first_name(self, value):
        self.first_name_encrypted = self.encrypt_field(value)
    
    def set_last_name(self, value):
        self.last_name_encrypted = self.encrypt_field(value)
    
    def set_email(self, value):
        self.email_encrypted = self.encrypt_field(value)
    
    def set_phone_number(self, value):
        self.phone_number_encrypted = self.encrypt_field(value) if value else b''
    
    def set_current_location(self, value):
        self.current_location_encrypted = self.encrypt_field(value)
    
    # Getter methods
    def get_first_name(self):
        return self.decrypt_field(self.first_name_encrypted)
    
    def get_last_name(self):
        return self.decrypt_field(self.last_name_encrypted)
    
    def get_email(self):
        return self.decrypt_field(self.email_encrypted)
    
    def get_phone_number(self):
        return self.decrypt_field(self.phone_number_encrypted)
    
    def get_current_location(self):
        return self.decrypt_field(self.current_location_encrypted)
    
    def get_full_name(self):
        return f"{self.get_first_name()} {self.get_last_name()}"
    
    # Keep backward compatibility properties
    @property
    def first_name(self):
        return self.get_first_name()
    
    @property
    def last_name(self):
        return self.get_last_name()
    
    @property
    def email(self):
        return self.get_email()
    
    @property
    def phone_number(self):
        return self.get_phone_number()
    
    @property
    def current_location(self):
        return self.get_current_location()
    
    def calculate_scores(self):
        """Calculate total score, percentage, and eligibility"""
        # Sum all criteria scores
        criteria_scores = [
            self.original_contributions,
            self.published_material,
            self.judging_work,
            self.scholarly_articles,
            self.leading_role,
            self.high_salary,
            self.exhibitions,
            self.membership,
            self.awards,
            self.commercial_success
        ]
        
        self.total_score = sum(criteria_scores)
        
        # Calculate percentage (max score is 30 - 10 criteria × 3 points each)
        self.percentage_score = (self.total_score / 30) * 100
        
        # Count criteria met (score >= 2 is considered "met")
        self.criteria_met = sum(1 for score in criteria_scores if score >= 2)
        
        # Determine eligibility status
        if self.criteria_met >= 3 and self.total_score >= 15:
            self.eligibility_status = 'Highly Eligible'
        elif self.criteria_met >= 3:
            self.eligibility_status = 'Eligible'
        elif self.criteria_met >= 2:
            self.eligibility_status = 'Potentially Eligible'
        else:
            self.eligibility_status = 'Not Eligible'
        
        self.save()
    
    def get_strength_level(self):
        """Return strength level based on percentage"""
        if self.percentage_score >= 75:
            return 'Excellent'
        elif self.percentage_score >= 60:
            return 'Strong'
        elif self.percentage_score >= 40:
            return 'Good'
        elif self.percentage_score >= 20:
            return 'Fair'
        else:
            return 'Weak'
    
    def get_recommendations(self):
        """Generate personalized recommendations"""
        recommendations = []
        
        criteria_map = {
            'original_contributions': 'Original Contributions',
            'published_material': 'Published Material',
            'judging_work': 'Judging Work',
            'scholarly_articles': 'Scholarly Articles',
            'leading_role': 'Leading Role',
            'high_salary': 'High Salary',
            'exhibitions': 'Exhibitions',
            'membership': 'Membership',
            'awards': 'Awards',
            'commercial_success': 'Commercial Success'
        }
        
        for field, name in criteria_map.items():
            score = getattr(self, field)
            if score < 2:
                recommendations.append(f"Strengthen your {name} evidence")
        
        return recommendations[:5]  # Return top 5 recommendations
