from django.shortcuts import render, redirect, get_object_or_404
from django.http import JsonResponse
from django.views.decorators.http import require_http_methods
from .models import ProfileAssessment
import json

def test_page(request):
    """Test page to verify app is working"""
    return render(request, 'assessment/test.html')

def assessment_form(request):
    """Display the assessment form"""
    return render(request, 'assessment/form_v2.html')

@require_http_methods(["POST"])
def submit_assessment(request):
    """Handle assessment form submission"""
    try:
        # Get form data
        data = request.POST
        
        # Create assessment instance
        assessment = ProfileAssessment()
        
        # Set encrypted personal information
        assessment.set_first_name(data.get('first_name'))
        assessment.set_last_name(data.get('last_name'))
        assessment.set_email(data.get('email'))
        assessment.set_phone_number(data.get('phone_number', ''))
        assessment.country_code = data.get('country_code', '+1')
        assessment.set_current_location(data.get('current_location'))
        assessment.field_of_expertise = data.get('field_of_expertise')
        assessment.years_of_experience = data.get('years_of_experience')
        
        # Set criteria scores
        assessment.original_contributions = int(data.get('original_contributions', 0))
        assessment.published_material = int(data.get('published_material', 0))
        assessment.judging_work = int(data.get('judging_work', 0))
        assessment.scholarly_articles = int(data.get('scholarly_articles', 0))
        assessment.leading_role = int(data.get('leading_role', 0))
        assessment.high_salary = int(data.get('high_salary', 0))
        assessment.exhibitions = int(data.get('exhibitions', 0))
        assessment.membership = int(data.get('membership', 0))
        assessment.awards = int(data.get('awards', 0))
        assessment.commercial_success = int(data.get('commercial_success', 0))
        
        # Save first to get an ID
        assessment.save()
        
        # Calculate scores
        assessment.calculate_scores()
        
        # Create Lead in CRM system
        try:
            from crm.models import Lead
            lead = Lead.objects.create(
                source='assessment',
                assessment=assessment,
                email=data.get('email'),
                first_name=data.get('first_name'),
                last_name=data.get('last_name'),
                phone=data.get('phone_number', ''),
                status='new'
            )
        except Exception as lead_error:
            # Log error but don't fail the assessment submission
            print(f"Lead creation error: {lead_error}")
        
        # Redirect to results page
        return redirect('assessment_results', assessment_id=assessment.assessment_id)
        
    except Exception as e:
        return JsonResponse({'error': str(e)}, status=400)

def assessment_results(request, assessment_id):
    """Display assessment results"""
    assessment = get_object_or_404(ProfileAssessment, assessment_id=assessment_id)
    
    # Get all criteria with scores and percentage
    criteria_data = [
        {
            'name': 'Original Contributions',
            'score': assessment.original_contributions,
            'percentage': round(assessment.original_contributions * 33.33, 2),
            'description': 'Original scientific, scholarly, artistic contributions'
        },
        {
            'name': 'Published Material',
            'score': assessment.published_material,
            'percentage': round(assessment.published_material * 33.33, 2),
            'description': 'Material published about you'
        },
        {
            'name': 'Judging Work',
            'score': assessment.judging_work,
            'percentage': round(assessment.judging_work * 33.33, 2),
            'description': 'Judging others\' work'
        },
        {
            'name': 'Scholarly Articles',
            'score': assessment.scholarly_articles,
            'percentage': round(assessment.scholarly_articles * 33.33, 2),
            'description': 'Authored scholarly articles'
        },
        {
            'name': 'Leading Role',
            'score': assessment.leading_role,
            'percentage': round(assessment.leading_role * 33.33, 2),
            'description': 'Leading role in organizations'
        },
        {
            'name': 'High Salary',
            'score': assessment.high_salary,
            'percentage': round(assessment.high_salary * 33.33, 2),
            'description': 'High salary/remuneration'
        },
        {
            'name': 'Exhibitions',
            'score': assessment.exhibitions,
            'percentage': round(assessment.exhibitions * 33.33, 2),
            'description': 'Exhibitions/showcases'
        },
        {
            'name': 'Membership',
            'score': assessment.membership,
            'percentage': round(assessment.membership * 33.33, 2),
            'description': 'Membership in associations'
        },
        {
            'name': 'Awards',
            'score': assessment.awards,
            'percentage': round(assessment.awards * 33.33, 2),
            'description': 'Awards & recognition'
        },
        {
            'name': 'Commercial Success',
            'score': assessment.commercial_success,
            'percentage': round(assessment.commercial_success * 33.33, 2),
            'description': 'Commercial success'
        }
    ]
    
    context = {
        'assessment': assessment,
        'criteria_data': criteria_data,
        'recommendations': assessment.get_recommendations(),
        'strength_level': assessment.get_strength_level()
    }
    
    return render(request, 'assessment/results_v3.html', context)

def assessment_dashboard(request):
    """Display user's assessment dashboard"""
    # Get email from session or query parameter
    email = request.GET.get('email', '')
    
    if email:
        assessments = ProfileAssessment.objects.filter(email=email).order_by('-created_at')
    else:
        assessments = []
    
    context = {
        'assessments': assessments,
        'email': email
    }
    
    return render(request, 'assessment/dashboard.html', context)
