"""
Clean up notifications with invalid links (deleted projects, leads, payments, etc.)
"""
import os
import django

os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'myproject.settings')
django.setup()

from crm.models import Notification, Project, Lead, Payment, Task
import re

def clean_invalid_notifications():
    """Remove notifications pointing to non-existent resources"""
    print("\n" + "="*60)
    print("CLEANING INVALID NOTIFICATIONS")
    print("="*60)
    
    all_notifications = Notification.objects.all()
    print(f"\n✓ Total notifications: {all_notifications.count()}")
    
    deleted_count = 0
    
    for notif in all_notifications:
        should_delete = False
        reason = ""
        
        # Check if link contains project UUID
        project_match = re.search(r'/projects/([a-f0-9-]+)/', notif.link)
        if project_match:
            project_id = project_match.group(1)
            if not Project.objects.filter(project_id=project_id).exists():
                reason = "Invalid project link"
                should_delete = True
        
        # Check if link contains lead UUID
        lead_match = re.search(r'/leads/([a-f0-9-]+)/', notif.link)
        if lead_match:
            lead_id = lead_match.group(1)
            if not Lead.objects.filter(lead_id=lead_id).exists():
                reason = "Invalid lead link"
                should_delete = True
        
        # Check if link contains payment UUID
        payment_match = re.search(r'/payments/([a-f0-9-]+)/', notif.link)
        if payment_match:
            payment_id = payment_match.group(1)
            if not Payment.objects.filter(payment_id=payment_id).exists():
                reason = "Invalid payment link"
                should_delete = True
        
        # Check if link contains task UUID
        task_match = re.search(r'/tasks/([a-f0-9-]+)/', notif.link)
        if task_match:
            task_id = task_match.group(1)
            if not Task.objects.filter(task_id=task_id).exists():
                reason = "Invalid task link"
                should_delete = True
        
        # Check for old admin URLs that should be crm-admin
        if '/admin/crm/' in notif.link and '/admin/' not in notif.link.replace('/admin/crm/', ''):
            reason = "Old admin URL format"
            should_delete = True
        
        if should_delete:
            print(f"\n⚠ {reason}: {notif.link}")
            print(f"  Notification: {notif.title}")
            print(f"  User: {notif.user.get_full_name() if notif.user else 'Unknown'}")
            notif.delete()
            deleted_count += 1
    
    print(f"\n{'='*60}")
    print(f"CLEANUP COMPLETE")
    print(f"{'='*60}")
    print(f"\n✓ Deleted {deleted_count} invalid notification(s)")
    print(f"✓ Remaining notifications: {Notification.objects.count()}")
    
    # Show remaining notifications
    remaining = Notification.objects.all()[:10]
    if remaining.exists():
        print(f"\n{'='*60}")
        print("REMAINING NOTIFICATIONS:")
        print(f"{'='*60}")
        for notif in remaining:
            print(f"\n{notif.title}")
            print(f"  User: {notif.user.get_full_name() if notif.user else 'Unknown'}")
            print(f"  Link: {notif.link}")
            print(f"  Read: {notif.is_read}")
    else:
        print("\n✓ No notifications remaining")

if __name__ == '__main__':
    clean_invalid_notifications()
