from django.contrib import admin
from django.contrib.auth.models import User
from .models import (
    UserProfile, Lead, Service, Project, Payment, 
    Task, TaskFile, ProjectChat, ActivityLog, Notification, ProjectFile
)

@admin.register(UserProfile)
class UserProfileAdmin(admin.ModelAdmin):
    list_display = ['user', 'role', 'get_email', 'created_at']
    list_filter = ['role', 'created_at']
    search_fields = ['user__username', 'user__email', 'user__first_name', 'user__last_name']
    readonly_fields = ['created_at', 'updated_at']
    
    def get_email(self, obj):
        return obj.user.email
    get_email.short_description = 'Email'

@admin.register(Lead)
class LeadAdmin(admin.ModelAdmin):
    list_display = ['lead_id', 'get_full_name', 'email', 'source', 'status', 'assigned_to', 'created_at']
    list_filter = ['source', 'status', 'created_at']
    search_fields = ['email', 'first_name', 'last_name']
    readonly_fields = ['lead_id', 'created_at', 'updated_at']
    
    fieldsets = (
        ('Lead Information', {
            'fields': ('lead_id', 'source', 'first_name', 'last_name', 'email', 'phone')
        }),
        ('Management', {
            'fields': ('status', 'assigned_to', 'assigned_by', 'assigned_at')
        }),
        ('Conversion', {
            'fields': ('converted_to_project', 'converted_at')
        }),
        ('Notes', {
            'fields': ('notes',)
        }),
        ('Timestamps', {
            'fields': ('created_at', 'updated_at')
        }),
    )

@admin.register(Service)
class ServiceAdmin(admin.ModelAdmin):
    list_display = ['name', 'category', 'price', 'is_active', 'created_at']
    list_filter = ['category', 'is_active']
    search_fields = ['name', 'description']
    readonly_fields = ['service_id', 'created_at', 'updated_at']

@admin.register(Project)
class ProjectAdmin(admin.ModelAdmin):
    list_display = ['title', 'client', 'status', 'crm', 'project_manager', 'created_at']
    list_filter = ['status', 'created_at']
    search_fields = ['title', 'client__username', 'client__email']
    readonly_fields = ['project_id', 'created_at', 'updated_at']
    
    fieldsets = (
        ('Project Information', {
            'fields': ('project_id', 'title', 'description', 'client', 'lead', 'service')
        }),
        ('Team', {
            'fields': ('crm', 'lead_manager', 'project_manager')
        }),
        ('Details', {
            'fields': ('status', 'budget', 'start_date', 'end_date')
        }),
        ('Timestamps', {
            'fields': ('created_at', 'updated_at')
        }),
    )

@admin.register(Payment)
class PaymentAdmin(admin.ModelAdmin):
    list_display = ['payment_id', 'project', 'client', 'amount', 'status', 'payment_date', 'created_at']
    list_filter = ['status', 'payment_method', 'created_at']
    search_fields = ['project__title', 'client__username']
    readonly_fields = ['payment_id', 'created_at', 'updated_at']

@admin.register(Task)
class TaskAdmin(admin.ModelAdmin):
    list_display = ['title', 'project', 'assigned_to', 'status', 'priority', 'due_date', 'created_at']
    list_filter = ['status', 'priority', 'created_at']
    search_fields = ['title', 'description', 'project__title']
    readonly_fields = ['task_id', 'created_at', 'updated_at']

@admin.register(TaskFile)
class TaskFileAdmin(admin.ModelAdmin):
    list_display = ['task', 'file', 'uploaded_by', 'uploaded_at']
    list_filter = ['uploaded_at']
    search_fields = ['task__title', 'description']
    readonly_fields = ['file_id', 'uploaded_at']

@admin.register(ProjectChat)
class ProjectChatAdmin(admin.ModelAdmin):
    list_display = ['project', 'sender', 'is_task_related', 'timestamp']
    list_filter = ['is_task_related', 'timestamp']
    search_fields = ['project__title', 'sender__username']
    readonly_fields = ['message_id', 'timestamp']

@admin.register(ActivityLog)
class ActivityLogAdmin(admin.ModelAdmin):
    list_display = ['user', 'action', 'model_name', 'timestamp']
    list_filter = ['action', 'model_name', 'timestamp']
    search_fields = ['user__username', 'description']
    readonly_fields = ['log_id', 'timestamp']

@admin.register(Notification)
class NotificationAdmin(admin.ModelAdmin):
    list_display = ['user', 'title', 'type', 'is_read', 'created_at']
    list_filter = ['type', 'is_read', 'created_at']
    search_fields = ['user__username', 'title', 'message']
    readonly_fields = ['notification_id', 'created_at']


@admin.register(ProjectFile)
class ProjectFileAdmin(admin.ModelAdmin):
    list_display = ['description', 'project', 'uploaded_by', 'uploaded_at']
    list_filter = ['uploaded_at', 'project']
    search_fields = ['description', 'project__title']
    readonly_fields = ['uploaded_at']
