from django.db import models
from django.contrib.auth.models import User
from django.core.validators import MinValueValidator
from cryptography.fernet import Fernet
from django.conf import settings
import base64
import hashlib
import uuid

# User Profile extending Django User
class UserProfile(models.Model):
    """Extended User profile with role-based access"""
    
    ROLE_CHOICES = [
        ('admin', 'Admin'),
        ('client', 'Client'),
        ('crm', 'CRM'),
        ('lead_manager', 'Lead Manager'),
        ('project_manager', 'Project Manager'),
        ('employee', 'Employee'),
    ]
    
    user = models.OneToOneField(User, on_delete=models.CASCADE, related_name='profile')
    role = models.CharField(max_length=20, choices=ROLE_CHOICES, default='client')
    phone_number_encrypted = models.BinaryField(null=True, blank=True)
    profile_picture = models.ImageField(upload_to='profiles/', null=True, blank=True)
    created_by = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, blank=True, related_name='created_profiles')
    
    # Staff approval fields (for self-registration)
    is_approved = models.BooleanField(default=True)  # Default True for backward compatibility
    approval_requested_at = models.DateTimeField(null=True, blank=True)
    approved_by = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, blank=True, related_name='approved_staff')
    approved_at = models.DateTimeField(null=True, blank=True)
    
    # Account verification fields (for clients)
    VERIFICATION_STATUS_CHOICES = [
        ('unverified', 'Unverified'),
        ('pending', 'Pending Verification'),
        ('verified', 'Verified'),
        ('rejected', 'Rejected'),
    ]
    verification_status = models.CharField(max_length=20, choices=VERIFICATION_STATUS_CHOICES, default='unverified')
    id_proof = models.FileField(upload_to='id_proofs/', null=True, blank=True)
    id_proof_uploaded_at = models.DateTimeField(null=True, blank=True)
    verified_by = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, blank=True, related_name='verified_profiles')
    verified_at = models.DateTimeField(null=True, blank=True)
    verification_notes = models.TextField(blank=True)
    
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    
    class Meta:
        db_table = 'crm_userprofile'
    
    @staticmethod
    def get_cipher():
        """Get encryption cipher"""
        key = hashlib.sha256(settings.SECRET_KEY.encode()).digest()
        key_base64 = base64.urlsafe_b64encode(key)
        return Fernet(key_base64)
    
    def set_phone_number(self, value):
        if value:
            cipher = self.get_cipher()
            self.phone_number_encrypted = cipher.encrypt(value.encode())
        else:
            self.phone_number_encrypted = b''
    
    def get_phone_number(self):
        if self.phone_number_encrypted:
            cipher = self.get_cipher()
            return cipher.decrypt(self.phone_number_encrypted).decode()
        return ''
    
    @property
    def phone_number(self):
        return self.get_phone_number()
    
    def __str__(self):
        return f"{self.user.get_full_name()} ({self.get_role_display()})"


class Lead(models.Model):
    """Lead model for capturing form submissions"""
    
    SOURCE_CHOICES = [
        ('contact', 'Contact Form'),
        ('assessment', 'Assessment Form'),
    ]
    
    STATUS_CHOICES = [
        ('new', 'New'),
        ('contacted', 'Contacted'),
        ('qualified', 'Qualified'),
        ('proposal_sent', 'Proposal Sent'),
        ('converted', 'Converted'),
        ('rejected', 'Rejected'),
    ]
    
    lead_id = models.UUIDField(default=uuid.uuid4, editable=False, unique=True)
    source = models.CharField(max_length=20, choices=SOURCE_CHOICES)
    
    # Link to form submissions
    contact = models.ForeignKey('main.Contact', on_delete=models.CASCADE, null=True, blank=True)
    assessment = models.ForeignKey('assessment.ProfileAssessment', on_delete=models.CASCADE, null=True, blank=True)
    
    # Lead details (cached from forms)
    email = models.EmailField()
    first_name = models.CharField(max_length=100)
    last_name = models.CharField(max_length=100)
    phone = models.CharField(max_length=20, blank=True)
    
    # Lead management
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='new')
    assigned_to = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, blank=True, related_name='assigned_leads')
    assigned_by = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, blank=True, related_name='assigned_leads_by')
    assigned_at = models.DateTimeField(null=True, blank=True)
    
    # Conversion
    converted_to_project = models.ForeignKey('Project', on_delete=models.SET_NULL, null=True, blank=True, related_name='source_lead')
    converted_at = models.DateTimeField(null=True, blank=True)
    
    # Notes
    notes = models.TextField(blank=True)
    
    # Timestamps
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    
    class Meta:
        ordering = ['-created_at']
        indexes = [
            models.Index(fields=['email']),
            models.Index(fields=['status']),
            models.Index(fields=['assigned_to']),
        ]
    
    def __str__(self):
        return f"{self.first_name} {self.last_name} - {self.get_status_display()}"
    
    def get_full_name(self):
        return f"{self.first_name} {self.last_name}"


class Service(models.Model):
    """Services that clients can purchase"""
    
    CATEGORY_CHOICES = [
        ('visa', 'Visa Services'),
        ('consultation', 'Consultation'),
        ('documentation', 'Documentation'),
        ('assessment', 'Profile Assessment'),
        ('other', 'Other'),
    ]
    
    service_id = models.UUIDField(default=uuid.uuid4, editable=False, unique=True)
    name = models.CharField(max_length=200)
    description = models.TextField()
    category = models.CharField(max_length=50, choices=CATEGORY_CHOICES)
    price = models.DecimalField(max_digits=10, decimal_places=2, validators=[MinValueValidator(0)])
    image = models.ImageField(upload_to='services/', null=True, blank=True)
    is_active = models.BooleanField(default=True)
    
    created_by = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, related_name='created_services')
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    
    class Meta:
        ordering = ['name']
    
    def __str__(self):
        return f"{self.name} - ${self.price}"


class Project(models.Model):
    """Project model for managing client work"""
    
    STATUS_CHOICES = [
        ('pending_payment', 'Pending Payment'),
        ('active', 'Active'),
        ('on_hold', 'On Hold'),
        ('completed', 'Completed'),
        ('closed', 'Closed'),
    ]
    
    project_id = models.UUIDField(default=uuid.uuid4, editable=False, unique=True)
    title = models.CharField(max_length=200)
    description = models.TextField()
    
    # Relationships
    client = models.ForeignKey(User, on_delete=models.CASCADE, null=True, blank=True, related_name='client_projects')
    client_email = models.EmailField(null=True, blank=True)  # Store email for projects without user accounts yet
    lead = models.ForeignKey(Lead, on_delete=models.SET_NULL, null=True, blank=True, related_name='projects')
    service = models.ForeignKey(Service, on_delete=models.SET_NULL, null=True, blank=True)
    
    # Team assignments
    crm = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, blank=True, related_name='crm_projects')
    lead_manager = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, blank=True, related_name='managed_projects')
    project_manager = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, blank=True, related_name='pm_projects')
    
    # Project details
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='pending_payment')
    budget = models.DecimalField(max_digits=10, decimal_places=2, null=True, blank=True)
    start_date = models.DateField(null=True, blank=True)
    end_date = models.DateField(null=True, blank=True)
    
    # Timestamps
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    
    class Meta:
        ordering = ['-created_at']
        indexes = [
            models.Index(fields=['client']),
            models.Index(fields=['status']),
            models.Index(fields=['crm']),
        ]
    
    def __str__(self):
        return f"{self.title} - {self.client.get_full_name()}"
    
    def get_progress_percentage(self):
        """Calculate project progress based on completed tasks"""
        total_tasks = self.tasks.count()
        if total_tasks == 0:
            return 0
        completed_tasks = self.tasks.filter(status='completed').count()
        return int((completed_tasks / total_tasks) * 100)


class Payment(models.Model):
    """Payment model for tracking client payments"""
    
    STATUS_CHOICES = [
        ('pending', 'Pending'),
        ('approved', 'Approved'),
        ('rejected', 'Rejected'),
    ]
    
    PAYMENT_METHOD_CHOICES = [
        ('bank_transfer', 'Bank Transfer'),
        ('credit_card', 'Credit Card'),
        ('paypal', 'PayPal'),
        ('other', 'Other'),
    ]
    
    PAYMENT_FROM_CHOICES = [
        ('india', 'India'),
        ('outside_india', 'Outside India'),
    ]
    
    payment_id = models.UUIDField(default=uuid.uuid4, editable=False, unique=True)
    project = models.ForeignKey(Project, on_delete=models.CASCADE, related_name='payments')
    client = models.ForeignKey(User, on_delete=models.CASCADE, related_name='payments')
    
    amount = models.DecimalField(max_digits=10, decimal_places=2, validators=[MinValueValidator(0)])
    transaction_id_encrypted = models.BinaryField()
    payment_method = models.CharField(max_length=20, choices=PAYMENT_METHOD_CHOICES)
    payment_from = models.CharField(max_length=20, choices=PAYMENT_FROM_CHOICES, default='india')
    payment_date = models.DateField()
    screenshot = models.ImageField(upload_to='payments/')
    
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='pending')
    approved_by = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, blank=True, related_name='approved_payments')
    approved_at = models.DateTimeField(null=True, blank=True)
    notes = models.TextField(blank=True)
    
    # Invoice fields
    invoice_number = models.CharField(max_length=50, unique=True, null=True, blank=True)
    invoice_generated_at = models.DateTimeField(null=True, blank=True)
    
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    
    class Meta:
        ordering = ['-created_at']
    
    @staticmethod
    def get_cipher():
        key = hashlib.sha256(settings.SECRET_KEY.encode()).digest()
        key_base64 = base64.urlsafe_b64encode(key)
        return Fernet(key_base64)
    
    def set_transaction_id(self, value):
        cipher = self.get_cipher()
        self.transaction_id_encrypted = cipher.encrypt(value.encode())
    
    def get_transaction_id(self):
        cipher = self.get_cipher()
        return cipher.decrypt(self.transaction_id_encrypted).decode()
    
    @property
    def transaction_id(self):
        return self.get_transaction_id()
    
    def generate_invoice_number(self):
        """Generate unique invoice number in format: INV-YYYY-NNNN"""
        from django.utils import timezone
        
        if self.invoice_number:
            return self.invoice_number
        
        # Get current year
        year = timezone.now().year
        
        # Get the last invoice number for this year
        last_payment = Payment.objects.filter(
            invoice_number__startswith=f'INV-{year}-'
        ).order_by('-invoice_number').first()
        
        if last_payment and last_payment.invoice_number:
            # Extract the sequence number
            last_seq = int(last_payment.invoice_number.split('-')[-1])
            new_seq = last_seq + 1
        else:
            new_seq = 1
        
        # Generate invoice number
        self.invoice_number = f'INV-{year}-{new_seq:04d}'
        self.invoice_generated_at = timezone.now()
        self.save()
        
        return self.invoice_number
    
    def __str__(self):
        return f"Payment {self.payment_id} - ${self.amount} - {self.get_status_display()}"


class Task(models.Model):
    """Task model for project task management"""
    
    STATUS_CHOICES = [
        ('pending', 'Pending'),
        ('in_progress', 'In Progress'),
        ('completed', 'Completed'),
    ]
    
    PRIORITY_CHOICES = [
        ('low', 'Low'),
        ('medium', 'Medium'),
        ('high', 'High'),
    ]
    
    task_id = models.UUIDField(default=uuid.uuid4, editable=False, unique=True)
    project = models.ForeignKey(Project, on_delete=models.CASCADE, related_name='tasks')
    title = models.CharField(max_length=200)
    description = models.TextField()
    
    assigned_to = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, related_name='assigned_tasks')
    assigned_by = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, related_name='created_tasks')
    
    status = models.CharField(max_length=20, choices=STATUS_CHOICES, default='pending')
    priority = models.CharField(max_length=10, choices=PRIORITY_CHOICES, default='medium')
    
    due_date = models.DateField(null=True, blank=True)
    completed_at = models.DateTimeField(null=True, blank=True)
    
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)
    
    class Meta:
        ordering = ['-created_at']
    
    def __str__(self):
        return f"{self.title} - {self.get_status_display()}"


class TaskFile(models.Model):
    """File uploads for tasks"""
    
    file_id = models.UUIDField(default=uuid.uuid4, editable=False, unique=True)
    task = models.ForeignKey(Task, on_delete=models.CASCADE, related_name='files')
    file = models.FileField(upload_to='task_files/')
    description = models.CharField(max_length=200, blank=True)
    uploaded_by = models.ForeignKey(User, on_delete=models.CASCADE)
    uploaded_at = models.DateTimeField(auto_now_add=True)
    
    def __str__(self):
        return f"{self.task.title} - {self.file.name}"


class ProjectChat(models.Model):
    """Chat messages for project communication"""
    
    message_id = models.UUIDField(default=uuid.uuid4, editable=False, unique=True)
    project = models.ForeignKey(Project, on_delete=models.CASCADE, related_name='chat_messages')
    sender = models.ForeignKey(User, on_delete=models.CASCADE)
    message_encrypted = models.BinaryField()
    
    is_task_related = models.BooleanField(default=False)
    task = models.ForeignKey(Task, on_delete=models.SET_NULL, null=True, blank=True, related_name='chat_messages')
    
    is_read = models.BooleanField(default=False)
    timestamp = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        ordering = ['timestamp']
    
    @staticmethod
    def get_cipher():
        key = hashlib.sha256(settings.SECRET_KEY.encode()).digest()
        key_base64 = base64.urlsafe_b64encode(key)
        return Fernet(key_base64)
    
    def set_message(self, value):
        cipher = self.get_cipher()
        self.message_encrypted = cipher.encrypt(value.encode())
    
    def get_message(self):
        cipher = self.get_cipher()
        return cipher.decrypt(self.message_encrypted).decode()
    
    @property
    def message(self):
        return self.get_message()
    
    def __str__(self):
        return f"{self.sender.username} - {self.timestamp}"


class ActivityLog(models.Model):
    """Activity logging for audit trail"""
    
    log_id = models.UUIDField(default=uuid.uuid4, editable=False, unique=True)
    user = models.ForeignKey(User, on_delete=models.SET_NULL, null=True)
    action = models.CharField(max_length=100)
    model_name = models.CharField(max_length=50)
    object_id = models.CharField(max_length=100)
    description = models.TextField()
    ip_address = models.GenericIPAddressField(null=True, blank=True)
    timestamp = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        ordering = ['-timestamp']
    
    def __str__(self):
        return f"{self.user} - {self.action} - {self.timestamp}"


class Notification(models.Model):
    """Notification model for user alerts"""
    
    TYPE_CHOICES = [
        ('lead_assigned', 'Lead Assigned'),
        ('task_assigned', 'Task Assigned'),
        ('payment_submitted', 'Payment Submitted'),
        ('payment_approved', 'Payment Approved'),
        ('payment_rejected', 'Payment Rejected'),
        ('project_assigned', 'Project Assigned'),
        ('project_status_changed', 'Project Status Changed'),
        ('task_completed', 'Task Completed'),
        ('new_message', 'New Message'),
    ]
    
    notification_id = models.UUIDField(default=uuid.uuid4, editable=False, unique=True)
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='notifications')
    title = models.CharField(max_length=200)
    message = models.TextField()
    type = models.CharField(max_length=30, choices=TYPE_CHOICES)
    link = models.CharField(max_length=200, blank=True)
    is_read = models.BooleanField(default=False)
    created_at = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        ordering = ['-created_at']
    
    def __str__(self):
        return f"{self.user.username} - {self.title}"


class ProjectFile(models.Model):
    """Files uploaded for projects (visible to client, CRM, lead manager, admin)"""
    
    project = models.ForeignKey(Project, on_delete=models.CASCADE, related_name='project_files')
    file = models.FileField(upload_to='project_files/')
    description = models.CharField(max_length=200)
    uploaded_by = models.ForeignKey(User, on_delete=models.CASCADE)
    uploaded_at = models.DateTimeField(auto_now_add=True)
    
    class Meta:
        ordering = ['-uploaded_at']
    
    def __str__(self):
        return f"{self.project.title} - {self.description}"
