from django.shortcuts import render, redirect, get_object_or_404
from django.contrib.auth import authenticate, login, logout
from django.contrib.auth.decorators import login_required
from django.contrib.auth.models import User
from django.http import JsonResponse
from django.views.decorators.http import require_http_methods
from django.views.decorators.csrf import csrf_exempt
from django.db.models import Q, Count, Sum
from django.utils import timezone
from datetime import datetime
import json

from .models import (
    UserProfile, Lead, Service, Project, Payment,
    Task, TaskFile, ProjectChat, ActivityLog, Notification, ProjectFile
)

# ============================================
# CLIENT AUTHENTICATION VIEWS
# ============================================

def client_login(request):
    """Client login page"""
    if request.method == 'POST':
        username = request.POST.get('username')
        password = request.POST.get('password')
        
        user = authenticate(request, username=username, password=password)
        
        if user is not None:
            # Check if user is a client
            try:
                profile = user.profile
                if profile.role == 'client':
                    login(request, user)
                    return redirect('client_dashboard')
                else:
                    return render(request, 'crm/client/login.html', {
                        'error': 'Invalid credentials. Please use staff login.'
                    })
            except UserProfile.DoesNotExist:
                return render(request, 'crm/client/login.html', {
                    'error': 'Invalid credentials.'
                })
        else:
            return render(request, 'crm/client/login.html', {
                'error': 'Invalid username or password.'
            })
    
    return render(request, 'crm/client/login.html')

def client_register(request):
    """Client registration page"""
    if request.method == 'POST':
        # Get form data
        username = request.POST.get('username')
        email = request.POST.get('email')
        password = request.POST.get('password')
        password_confirm = request.POST.get('password_confirm')
        first_name = request.POST.get('first_name')
        last_name = request.POST.get('last_name')
        phone = request.POST.get('phone', '')
        
        # Validation
        if password != password_confirm:
            return render(request, 'crm/client/register.html', {
                'error': 'Passwords do not match.'
            })
        
        if User.objects.filter(username=username).exists():
            return render(request, 'crm/client/register.html', {
                'error': 'Username already exists.'
            })
        
        if User.objects.filter(email=email).exists():
            return render(request, 'crm/client/register.html', {
                'error': 'Email already registered.'
            })
        
        # Create user
        user = User.objects.create_user(
            username=username,
            email=email,
            password=password,
            first_name=first_name,
            last_name=last_name
        )
        
        # Create profile
        profile = UserProfile.objects.create(
            user=user,
            role='client'
        )
        if phone:
            profile.set_phone_number(phone)
            profile.save()
        
        # Link any existing projects with this email (from lead conversions)
        existing_projects = Project.objects.filter(
            client_email__iexact=email,  # Case-insensitive match
            client__isnull=True  # Only projects without a client user yet
        )
        
        if existing_projects.exists():
            # Link all matching projects to this user
            for project in existing_projects:
                project.client = user
                project.save()
                
                # Notify user about existing project
                Notification.objects.create(
                    user=user,
                    title='Project Found',
                    message=f'We found an existing project for you: {project.title}. Please submit payment to activate.',
                    type='project_assigned',
                    link='/client/payments/'
                )
            
            # Log activity
            ActivityLog.objects.create(
                user=user,
                action='Projects Linked',
                model_name='Project',
                object_id=str(user.id),
                description=f'{existing_projects.count()} existing project(s) linked to new client account'
            )
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action='User Registration',
            model_name='User',
            object_id=str(user.id),
            description=f'New client registered: {user.get_full_name()}'
        )
        
        # Auto-login
        login(request, user)
        
        return redirect('client_dashboard')
    
    return render(request, 'crm/client/register.html')

@login_required
def client_logout(request):
    """Client logout"""
    logout(request)
    return redirect('client_login')

# ============================================
# CLIENT DASHBOARD VIEWS
# ============================================

@login_required
def client_dashboard(request):
    """Client dashboard - overview"""
    user = request.user
    
    # Check if user is client
    if not hasattr(user, 'profile') or user.profile.role != 'client':
        return redirect('staff_login')
    
    # Get user's projects
    projects = Project.objects.filter(client=user).order_by('-created_at')
    
    # Get recent notifications
    notifications = Notification.objects.filter(user=user, is_read=False)[:5]
    
    # Statistics
    stats = {
        'total_projects': projects.count(),
        'active_projects': projects.filter(status='active').count(),
        'completed_projects': projects.filter(status='completed').count(),
        'pending_payments': Payment.objects.filter(client=user, status='pending').count()
    }
    
    context = {
        'user': user,
        'projects': projects[:5],  # Show recent 5
        'notifications': notifications,
        'stats': stats
    }
    
    return render(request, 'crm/client/dashboard.html', context)

@login_required
def client_profile(request):
    """Client profile page - view and edit profile"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'client':
        return redirect('staff_login')
    
    success = None
    error = None
    
    if request.method == 'POST':
        form_type = request.POST.get('form_type')
        
        # Handle Personal Information Update
        if form_type == 'personal_info':
            try:
                user.first_name = request.POST.get('first_name')
                user.last_name = request.POST.get('last_name')
                user.email = request.POST.get('email')
                user.save()
                
                # Update phone number
                phone = request.POST.get('phone', '')
                if phone:
                    user.profile.set_phone_number(phone)
                else:
                    user.profile.phone_number_encrypted = b''
                user.profile.save()
                
                success = 'Personal information updated successfully!'
            except Exception as e:
                error = f'Error updating profile: {str(e)}'
        
        # Handle Password Change
        elif form_type == 'change_password':
            new_password = request.POST.get('new_password')
            confirm_password = request.POST.get('confirm_password')
            
            if new_password != confirm_password:
                error = 'Passwords do not match!'
            elif len(new_password) < 8:
                error = 'Password must be at least 8 characters long!'
            else:
                try:
                    user.set_password(new_password)
                    user.save()
                    success = 'Password updated successfully! Please login again with your new password.'
                except Exception as e:
                    error = f'Error updating password: {str(e)}'
        
        # Handle ID Proof Upload
        elif form_type == 'upload_id':
            id_proof = request.FILES.get('id_proof')
            
            if id_proof:
                # Check file size (5MB limit)
                if id_proof.size > 5 * 1024 * 1024:
                    error = 'File size must be less than 5MB!'
                else:
                    try:
                        user.profile.id_proof = id_proof
                        user.profile.verification_status = 'pending'
                        user.profile.id_proof_uploaded_at = timezone.now()
                        user.profile.save()
                        
                        # Notify admins
                        admin_users = User.objects.filter(is_superuser=True)
                        for admin in admin_users:
                            Notification.objects.create(
                                user=admin,
                                title='New ID Verification Request',
                                message=f'{user.get_full_name()} has submitted ID proof for verification.',
                                type='project_assigned',
                                link=f'/crm-admin/users/{user.id}/edit/'
                            )
                        
                        # Log activity
                        ActivityLog.objects.create(
                            user=user,
                            action='ID Proof Uploaded',
                            model_name='UserProfile',
                            object_id=str(user.profile.id),
                            description=f'Client uploaded ID proof for verification'
                        )
                        
                        success = 'ID proof uploaded successfully! Your account is now pending verification.'
                    except Exception as e:
                        error = f'Error uploading ID proof: {str(e)}'
            else:
                error = 'Please select a file to upload!'
    
    # Get stats for profile card
    projects_count = Project.objects.filter(client=user).count()
    approved_payments_count = Payment.objects.filter(client=user, status='approved').count()
    
    context = {
        'user': user,
        'success': success,
        'error': error,
        'projects_count': projects_count,
        'approved_payments_count': approved_payments_count,
    }
    
    return render(request, 'crm/client/profile.html', context)

@login_required
def client_leads(request):
    """Client leads page - view all submitted leads"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'client':
        return redirect('staff_login')
    
    # Get all leads matched by email
    leads = Lead.objects.filter(email=user.email).order_by('-created_at')
    
    context = {
        'user': user,
        'leads': leads
    }
    
    return render(request, 'crm/client/leads.html', context)

@login_required
def client_projects(request):
    """Client projects page - view all projects"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'client':
        return redirect('staff_login')
    
    # Get all projects
    projects = Project.objects.filter(client=user).order_by('-created_at')
    
    context = {
        'user': user,
        'projects': projects
    }
    
    return render(request, 'crm/client/projects.html', context)

@login_required
def client_project_detail(request, project_id):
    """Client project detail page"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'client':
        return redirect('staff_login')
    
    # Get project (ensure it belongs to this client)
    project = get_object_or_404(Project, project_id=project_id, client=user)
    
    # Get project tasks
    tasks = Task.objects.filter(project=project).order_by('-created_at')
    
    # Get project files
    project_files = ProjectFile.objects.filter(project=project).order_by('-uploaded_at')
    
    # Get project chat messages
    chat_messages = ProjectChat.objects.filter(project=project).order_by('timestamp')
    
    context = {
        'user': user,
        'project': project,
        'tasks': tasks,
        'project_files': project_files,
        'chat_messages': chat_messages
    }
    
    return render(request, 'crm/client/project_detail.html', context)

@login_required
@require_http_methods(["POST"])
def client_send_message(request, project_id):
    """Client send message to project team"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'client':
        return redirect('staff_login')
    
    # Get project (ensure it belongs to this client)
    project = get_object_or_404(Project, project_id=project_id, client=user)
    
    # Get message content
    message_text = request.POST.get('message', '').strip()
    
    if message_text:
        # Create chat message
        chat_message = ProjectChat()
        chat_message.project = project
        chat_message.sender = user
        chat_message.set_message(message_text)
        chat_message.save()
        
        # Create notifications for all project participants
        participants = []
        
        # Add client
        if project.client:
            participants.append(project.client)
        
        # Add CRM staff
        if project.crm:
            participants.append(project.crm)
        
        # Add Project Manager
        if project.project_manager:
            participants.append(project.project_manager)
        
        # Send notification to everyone except the sender
        for participant in participants:
            if participant != user:
                # Determine the correct link based on role
                if hasattr(participant, 'profile'):
                    if participant.profile.role == 'client':
                        link = f'/client/projects/{project.project_id}/'
                    elif participant.profile.role == 'project_manager':
                        link = f'/pm/projects/{project.project_id}/'
                    elif participant.profile.role == 'crm':
                        link = f'/crm/projects/{project.project_id}/'
                    else:
                        link = f'/client/projects/{project.project_id}/'
                else:
                    link = f'/client/projects/{project.project_id}/'
                
                Notification.objects.create(
                    user=participant,
                    title='New Project Message',
                    message=f'{user.get_full_name()} sent a message in {project.title}',
                    type='message',
                    link=link
                )
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action='Message Sent',
            model_name='ProjectChat',
            object_id=str(chat_message.message_id),
            description=f'Client sent message in project: {project.title}'
        )
    
    return redirect('client_project_detail', project_id=project_id)

@login_required
def client_services(request):
    """Client services page - browse and purchase services"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'client':
        return redirect('staff_login')
    
    # Get active services
    services = Service.objects.filter(is_active=True).order_by('category', 'name')
    
    context = {
        'user': user,
        'services': services
    }
    
    return render(request, 'crm/client/services.html', context)

@login_required
@login_required
@require_http_methods(["POST"])
def purchase_service(request, service_id):
    """Handle service purchase"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'client':
        return redirect('staff_login')
    
    service = get_object_or_404(Service, service_id=service_id)
    
    # Check if user already has a pending payment project for this service
    existing_pending = Project.objects.filter(
        client=user,
        service=service,
        status='pending_payment'
    ).first()
    
    if existing_pending:
        # Redirect to payments page if already purchased
        return redirect('client_payments')
    
    # Create project from service purchase
    project = Project.objects.create(
        title=f"{service.name} - {user.get_full_name()}",
        description=f"Service purchased: {service.description}",
        client=user,
        service=service,
        status='pending_payment',
        budget=service.price
    )
    
    # Note: Payment record will be created when client submits payment details
    
    # Create notification for admin
    admin_users = User.objects.filter(is_superuser=True)
    for admin in admin_users:
        Notification.objects.create(
            user=admin,
            title='New Service Purchase',
            message=f'{user.get_full_name()} purchased {service.name}',
            type='project_assigned',
            link=f'/crm-admin/projects/{project.project_id}/'
        )
    
    # Log activity
    ActivityLog.objects.create(
        user=user,
        action='Service Purchase',
        model_name='Project',
        object_id=str(project.project_id),
        description=f'Client purchased service: {service.name}'
    )
    
    # Redirect to payments page with success message
    return redirect('client_payments')

@login_required
def client_payments(request):
    """Client payments page - view and submit payments"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'client':
        return redirect('staff_login')
    
    # Get all payments
    payments = Payment.objects.filter(client=user).order_by('-created_at')
    
    # Get projects pending payment that don't have a payment submitted yet
    pending_projects = Project.objects.filter(
        client=user,
        status='pending_payment'
    ).exclude(
        payments__isnull=False  # Exclude projects that already have payments
    )
    
    context = {
        'user': user,
        'payments': payments,
        'pending_projects': pending_projects
    }
    
    return render(request, 'crm/client/payments.html', context)

@login_required
@login_required
@require_http_methods(["POST"])
def submit_payment(request):
    """Handle payment submission"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'client':
        return redirect('staff_login')
    
    try:
        project_id = request.POST.get('project_id')
        amount = request.POST.get('amount')
        transaction_id = request.POST.get('transaction_id')
        payment_method = request.POST.get('payment_method')
        payment_from = request.POST.get('payment_from')
        payment_date = request.POST.get('payment_date')
        screenshot = request.FILES.get('screenshot')
        
        project = get_object_or_404(Project, project_id=project_id, client=user)
        
        # Create payment
        payment = Payment.objects.create(
            project=project,
            client=user,
            amount=amount,
            payment_method=payment_method,
            payment_from=payment_from,
            payment_date=payment_date,
            screenshot=screenshot,
            status='pending'
        )
        payment.set_transaction_id(transaction_id)
        payment.save()
        
        # Notify admin and CRM
        notify_users = User.objects.filter(is_superuser=True)
        if project.crm:
            notify_users = notify_users | User.objects.filter(id=project.crm.id)
        
        for notify_user in notify_users:
            Notification.objects.create(
                user=notify_user,
                title='New Payment Submitted',
                message=f'{user.get_full_name()} submitted payment for {project.title}',
                type='payment_submitted',
                link=f'/crm-admin/payments/'
            )
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action='Payment Submitted',
            model_name='Payment',
            object_id=str(payment.payment_id),
            description=f'Payment submitted for project: {project.title}'
        )
        
        # Redirect back to payments page
        return redirect('client_payments')
        
    except Exception as e:
        # In case of error, redirect back with error (you could use Django messages framework here)
        return redirect('client_payments')

@login_required
def view_invoice(request, payment_id):
    """View invoice for approved payment"""
    user = request.user
    
    payment = get_object_or_404(Payment, payment_id=payment_id)
    
    # Check permissions
    is_client = hasattr(user, 'profile') and user.profile.role == 'client' and payment.client == user
    is_crm = hasattr(user, 'profile') and user.profile.role == 'crm' and payment.project.crm == user
    is_admin = user.is_superuser
    
    if not (is_client or is_crm or is_admin):
        return redirect('staff_login')
    
    # Only show invoice for approved payments
    if payment.status != 'approved' or not payment.invoice_number:
        return redirect('client_payments' if is_client else 'admin_payments')
    
    context = {
        'user': user,
        'payment': payment,
        'project': payment.project,
        'client': payment.client,
    }
    
    return render(request, 'crm/invoice.html', context)

# ============================================
# STAFF AUTHENTICATION VIEWS
# ============================================

def staff_login(request):
    """Staff login page with role-based redirect"""
    if request.method == 'POST':
        username = request.POST.get('username')
        password = request.POST.get('password')
        
        user = authenticate(request, username=username, password=password)
        
        if user is not None:
            try:
                profile = user.profile
                
                # Redirect based on role
                if profile.role == 'client':
                    return render(request, 'crm/staff/login.html', {
                        'error': 'Please use client login.'
                    })
                
                # Check if staff account is approved (only for non-admin staff)
                if not user.is_superuser and profile.role != 'admin':
                    if not profile.is_approved:
                        return render(request, 'crm/staff/login.html', {
                            'error': 'Your account is pending admin approval. Please wait for approval before logging in.'
                        })
                
                login(request, user)
                
                # Role-based redirect
                if user.is_superuser or profile.role == 'admin':
                    return redirect('admin_dashboard')
                elif profile.role == 'crm':
                    return redirect('crm_dashboard')
                elif profile.role == 'lead_manager':
                    return redirect('lead_manager_dashboard')
                elif profile.role == 'project_manager':
                    return redirect('pm_dashboard')
                elif profile.role == 'employee':
                    return redirect('employee_dashboard')
                else:
                    return redirect('staff_dashboard')
                    
            except UserProfile.DoesNotExist:
                return render(request, 'crm/staff/login.html', {
                    'error': 'Invalid credentials.'
                })
        else:
            return render(request, 'crm/staff/login.html', {
                'error': 'Invalid username or password.'
            })
    
    return render(request, 'crm/staff/login.html')

def staff_register(request):
    """Staff self-registration page (requires admin approval)"""
    if request.method == 'POST':
        # Get form data
        username = request.POST.get('username')
        email = request.POST.get('email')
        password = request.POST.get('password')
        password_confirm = request.POST.get('password_confirm')
        first_name = request.POST.get('first_name')
        last_name = request.POST.get('last_name')
        phone = request.POST.get('phone', '')
        role = request.POST.get('role')
        
        # Validation
        if password != password_confirm:
            return render(request, 'crm/staff/register.html', {
                'error': 'Passwords do not match.'
            })
        
        # Validate role (only allow staff roles, not admin)
        allowed_roles = ['crm', 'lead_manager', 'project_manager', 'employee']
        if role not in allowed_roles:
            return render(request, 'crm/staff/register.html', {
                'error': 'Invalid role selected.'
            })
        
        if User.objects.filter(username=username).exists():
            return render(request, 'crm/staff/register.html', {
                'error': 'Username already exists.'
            })
        
        if User.objects.filter(email=email).exists():
            return render(request, 'crm/staff/register.html', {
                'error': 'Email already registered.'
            })
        
        # Create user (inactive until approved)
        user = User.objects.create_user(
            username=username,
            email=email,
            password=password,
            first_name=first_name,
            last_name=last_name,
            is_active=True  # Keep active but check approval in login
        )
        
        # Create profile with approval pending
        profile = UserProfile.objects.create(
            user=user,
            role=role,
            is_approved=False,  # Requires admin approval
            approval_requested_at=timezone.now()
        )
        if phone:
            profile.set_phone_number(phone)
            profile.save()
        
        # Notify all admins about pending approval
        admin_users = User.objects.filter(is_superuser=True)
        for admin in admin_users:
            Notification.objects.create(
                user=admin,
                title='New Staff Registration Pending',
                message=f'{user.get_full_name()} ({user.email}) has registered as {profile.get_role_display()} and is awaiting approval.',
                type='project_assigned',  # Using existing type
                link='/crm-admin/users/'
            )
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action='Staff Registration',
            model_name='User',
            object_id=str(user.id),
            description=f'New staff registered (pending approval): {user.get_full_name()} - {profile.get_role_display()}'
        )
        
        # Show success message
        return render(request, 'crm/staff/register.html', {
            'success': 'Registration successful! Your account is pending admin approval. You will be able to login once approved.'
        })
    
    return render(request, 'crm/staff/register.html')

@login_required
def staff_logout(request):
    """Staff logout"""
    logout(request)
    return redirect('staff_login')

# ============================================
# ADMIN DASHBOARD VIEWS
# ============================================

@login_required
def admin_dashboard(request):
    """Admin dashboard - overview and analytics"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    # Statistics
    stats = {
        'total_leads': Lead.objects.count(),
        'new_leads': Lead.objects.filter(status='new').count(),
        'converted_leads': Lead.objects.filter(status='converted').count(),
        'total_projects': Project.objects.count(),
        'active_projects': Project.objects.filter(status='active').count(),
        'pending_payments': Payment.objects.filter(status='pending').count(),
        'total_revenue': Payment.objects.filter(status='approved').aggregate(Sum('amount'))['amount__sum'] or 0,
        'total_clients': User.objects.filter(profile__role='client').count(),
        'crm_count': User.objects.filter(profile__role='crm').count(),
        'lead_manager_count': User.objects.filter(profile__role='lead_manager').count(),
        'project_manager_count': User.objects.filter(profile__role='project_manager').count(),
        'employee_count': User.objects.filter(profile__role='employee').count(),
        'pending_staff_approvals': User.objects.filter(
            is_superuser=False,
            profile__is_approved=False,
            profile__role__in=['crm', 'lead_manager', 'project_manager', 'employee']
        ).count(),
        'pending_verifications': UserProfile.objects.filter(
            role='client',
            verification_status='pending'
        ).count(),
    }
    
    # Recent leads
    recent_leads = Lead.objects.all().order_by('-created_at')[:10]
    
    # Recent projects
    recent_projects = Project.objects.all().order_by('-created_at')[:10]
    
    # Pending payments
    pending_payments = Payment.objects.filter(status='pending').order_by('-created_at')[:10]
    
    # Pending verifications
    pending_verifications = User.objects.filter(
        profile__role='client',
        profile__verification_status='pending'
    ).select_related('profile').order_by('-profile__id_proof_uploaded_at')[:10]
    
    # Recent notifications
    notifications = Notification.objects.filter(user=user, is_read=False)[:5]
    
    context = {
        'user': user,
        'stats': stats,
        'recent_leads': recent_leads,
        'recent_projects': recent_projects,
        'pending_payments': pending_payments,
        'pending_verifications': pending_verifications,
        'notifications': notifications
    }
    
    return render(request, 'crm/admin/dashboard.html', context)

@login_required
def admin_leads(request):
    """Admin leads management page"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    # Get all leads
    leads = Lead.objects.all().order_by('-created_at')
    
    # Apply filters
    status_filter = request.GET.get('status')
    source_filter = request.GET.get('source')
    search_query = request.GET.get('search')
    
    if status_filter:
        leads = leads.filter(status=status_filter)
    
    if source_filter:
        leads = leads.filter(source=source_filter)
    
    if search_query:
        leads = leads.filter(
            Q(first_name__icontains=search_query) |
            Q(last_name__icontains=search_query) |
            Q(email__icontains=search_query)
        )
    
    # Get all lead managers for assignment
    lead_managers = User.objects.filter(profile__role='lead_manager')
    
    context = {
        'user': user,
        'leads': leads,
        'lead_managers': lead_managers
    }
    
    return render(request, 'crm/admin/leads.html', context)

@login_required
def admin_projects(request):
    """Admin projects management page"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    # Get all projects
    projects = Project.objects.all().order_by('-created_at')
    
    # Apply filters
    status_filter = request.GET.get('status')
    search_query = request.GET.get('search')
    
    if status_filter:
        projects = projects.filter(status=status_filter)
    
    if search_query:
        projects = projects.filter(
            Q(title__icontains=search_query) |
            Q(client__first_name__icontains=search_query) |
            Q(client__last_name__icontains=search_query) |
            Q(client__email__icontains=search_query)
        )
    
    # Get clients and services for project creation
    clients = User.objects.filter(profile__role='client').order_by('first_name')
    services = Service.objects.filter(is_active=True).order_by('name')
    crm_staff = User.objects.filter(profile__role='crm').order_by('first_name')
    
    context = {
        'user': user,
        'projects': projects,
        'clients': clients,
        'services': services,
        'crm_staff': crm_staff
    }
    
    return render(request, 'crm/admin/projects.html', context)

@login_required
@require_http_methods(["POST"])
def admin_create_project(request):
    """Admin create new project"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    # Get form data
    client_id = request.POST.get('client_id')
    service_id = request.POST.get('service_id')
    crm_id = request.POST.get('crm_id')
    title = request.POST.get('title')
    description = request.POST.get('description')
    budget = request.POST.get('budget')
    
    # Validate required fields
    if not all([client_id, service_id, title, budget]):
        return redirect('admin_projects')
    
    # Get objects
    client = get_object_or_404(User, id=client_id)
    service = get_object_or_404(Service, service_id=service_id)
    
    # Create project
    project = Project.objects.create(
        client=client,
        service=service,
        title=title,
        description=description or f"Project for {service.name}",
        budget=budget,
        status='pending_payment'
    )
    
    # Assign CRM if selected
    if crm_id:
        crm_user = get_object_or_404(User, id=crm_id)
        project.crm = crm_user
        project.save()
        
        # Create notification for CRM
        Notification.objects.create(
            user=crm_user,
            title='New Project Assigned',
            message=f'You have been assigned a new project: {project.title}',
            type='project_assigned',
            link=f'/crm/projects/{project.project_id}/'
        )
    
    # Create notification for client
    Notification.objects.create(
        user=client,
        title='New Project Created',
        message=f'A new project has been created for you: {project.title}. Please submit payment to activate.',
        type='project_created',
        link=f'/client/payments/'
    )
    
    # Log activity
    ActivityLog.objects.create(
        user=user,
        action='Project Created',
        model_name='Project',
        object_id=str(project.project_id),
        description=f'Project created by admin: {project.title}'
    )
    
    return redirect('admin_project_detail', project_id=project.project_id)

@login_required
def admin_project_detail(request, project_id):
    """Admin project detail view"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    project = get_object_or_404(Project, project_id=project_id)
    
    # Get all CRM staff and Project Managers for assignment
    crm_staff = User.objects.filter(profile__role='crm')
    project_managers = User.objects.filter(profile__role='project_manager')
    
    # Get project tasks
    tasks = Task.objects.filter(project=project).order_by('-created_at')
    
    # Get project files
    project_files = ProjectFile.objects.filter(project=project).order_by('-uploaded_at')
    
    context = {
        'user': user,
        'project': project,
        'crm_staff': crm_staff,
        'project_managers': project_managers,
        'tasks': tasks,
        'project_files': project_files
    }
    
    return render(request, 'crm/admin/project_detail.html', context)

@login_required
@require_http_methods(["POST"])
def admin_assign_crm(request, project_id):
    """Admin assign CRM staff to project"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    project = get_object_or_404(Project, project_id=project_id)
    crm_id = request.POST.get('crm_id')
    
    if crm_id:
        crm_user = get_object_or_404(User, id=crm_id)
        project.crm = crm_user
        project.save()
        
        # Create notification for CRM
        Notification.objects.create(
            user=crm_user,
            title='New Project Assigned',
            message=f'You have been assigned a new project: {project.title}',
            type='project_assigned',
            link=f'/crm/projects/{project.project_id}/'
        )
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action='CRM Staff Assigned',
            model_name='Project',
            object_id=str(project.project_id),
            description=f'CRM Staff assigned: {crm_user.get_full_name()}'
        )
    
    return redirect('admin_project_detail', project_id=project_id)

@login_required
@require_http_methods(["POST"])
def admin_assign_pm(request, project_id):
    """Admin assign project manager to project"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    project = get_object_or_404(Project, project_id=project_id)
    pm_id = request.POST.get('pm_id')
    
    if pm_id:
        pm_user = get_object_or_404(User, id=pm_id)
        project.project_manager = pm_user
        project.save()
        
        # Create notification for Project Manager
        Notification.objects.create(
            user=pm_user,
            title='New Project Assigned',
            message=f'You have been assigned a new project: {project.title}',
            type='project_assigned',
            link=f'/pm/projects/{project.project_id}/'
        )
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action='Project Manager Assigned',
            model_name='Project',
            object_id=str(project.project_id),
            description=f'Project Manager assigned: {pm_user.get_full_name()}'
        )
    
    return redirect('admin_project_detail', project_id=project_id)

@login_required
def admin_payments(request):
    """Admin payments management page"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    # Get all payments
    payments = Payment.objects.all().order_by('-created_at')
    
    # Apply filters
    status_filter = request.GET.get('status')
    
    if status_filter:
        payments = payments.filter(status=status_filter)
    
    context = {
        'user': user,
        'payments': payments
    }
    
    return render(request, 'crm/admin/payments.html', context)

@login_required
def admin_services(request):
    """Admin services management page"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    # Get all services
    services = Service.objects.all().order_by('name')
    
    context = {
        'user': user,
        'services': services
    }
    
    return render(request, 'crm/admin/services.html', context)

@login_required
def admin_service_add(request):
    """Admin add service page"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    if request.method == 'POST':
        name = request.POST.get('name')
        description = request.POST.get('description')
        category = request.POST.get('category')
        price = request.POST.get('price')
        is_active = request.POST.get('is_active') == 'on'
        
        Service.objects.create(
            name=name,
            description=description,
            category=category,
            price=price,
            is_active=is_active
        )
        
        return redirect('admin_services')
    
    context = {
        'user': user
    }
    
    return render(request, 'crm/admin/service_add.html', context)

@login_required
def admin_service_edit(request, service_id):
    """Admin edit service page"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    service = get_object_or_404(Service, service_id=service_id)
    
    if request.method == 'POST':
        service.name = request.POST.get('name')
        service.description = request.POST.get('description')
        service.category = request.POST.get('category')
        service.price = request.POST.get('price')
        service.is_active = request.POST.get('is_active') == 'on'
        service.save()
        
        return redirect('admin_services')
    
    context = {
        'user': user,
        'service': service
    }
    
    return render(request, 'crm/admin/service_edit.html', context)

@login_required
@require_http_methods(["POST"])
def admin_service_toggle(request, service_id):
    """Admin toggle service active status"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    service = get_object_or_404(Service, service_id=service_id)
    service.is_active = not service.is_active
    service.save()
    
    return redirect('admin_services')

@login_required
def admin_lead_detail(request, lead_id):
    """Admin lead detail page"""
    user = request.user
    
    # Allow both superusers and assigned lead managers to view
    lead = get_object_or_404(Lead, lead_id=lead_id)
    
    if not user.is_superuser:
        if not hasattr(user, 'profile') or user.profile.role != 'lead_manager':
            return redirect('staff_login')
        if lead.assigned_to != user:
            return redirect('lead_manager_dashboard')
    
    # Get all lead managers for assignment dropdown (admin only)
    lead_managers = User.objects.filter(profile__role='lead_manager')
    
    # Get all CRM staff for project assignment (lead manager only, when converted)
    crm_staff_list = User.objects.filter(profile__role='crm')
    
    context = {
        'user': user,
        'lead': lead,
        'lead_managers': lead_managers,
        'crm_staff_list': crm_staff_list
    }
    
    # Use different template based on user role
    if user.is_superuser:
        return render(request, 'crm/admin/lead_detail.html', context)
    else:
        return render(request, 'crm/lead_manager/lead_detail.html', context)

@login_required
@require_http_methods(["POST"])
def admin_lead_assign(request, lead_id):
    """Assign lead to a lead manager"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    lead = get_object_or_404(Lead, lead_id=lead_id)
    assigned_to_id = request.POST.get('assigned_to')
    
    if assigned_to_id:
        assigned_user = get_object_or_404(User, id=assigned_to_id)
        lead.assigned_to = assigned_user
        lead.assigned_by = user
        lead.assigned_at = timezone.now()
        lead.save()
        
        # Create notification for assigned lead manager
        Notification.objects.create(
            user=assigned_user,
            title='New Lead Assigned',
            message=f'You have been assigned a new lead: {lead.get_full_name()}',
            type='lead_assigned',
            link=f'/crm-admin/leads/{lead.lead_id}/'
        )
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action='Lead Assigned',
            model_name='Lead',
            object_id=str(lead.lead_id),
            description=f'Lead assigned to {assigned_user.get_full_name()}'
        )
    
    return redirect('admin_lead_detail', lead_id=lead_id)

@login_required
@require_http_methods(["POST"])
def admin_lead_assign_crm(request, lead_id):
    """Assign converted lead/project to CRM staff (Lead Manager only)"""
    user = request.user
    lead = get_object_or_404(Lead, lead_id=lead_id)
    
    # Check permissions: must be assigned lead manager and lead must be converted
    if not hasattr(user, 'profile') or user.profile.role != 'lead_manager':
        return redirect('staff_login')
    if lead.assigned_to != user:
        return redirect('lead_manager_dashboard')
    if lead.status != 'converted':
        return redirect('admin_lead_detail', lead_id=lead_id)
    
    crm_assigned_to_id = request.POST.get('crm_assigned_to')
    
    if crm_assigned_to_id and lead.converted_to_project:
        crm_user = get_object_or_404(User, id=crm_assigned_to_id)
        project = lead.converted_to_project
        project.crm = crm_user
        project.save()
        
        # Create notification for CRM staff
        Notification.objects.create(
            user=crm_user,
            title='New Project Assigned',
            message=f'You have been assigned a new project: {project.title}',
            type='project_assigned',
            link=f'/crm-admin/projects/{project.project_id}/'
        )
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action='Project Assigned to CRM',
            model_name='Project',
            object_id=str(project.project_id),
            description=f'Project assigned to CRM staff: {crm_user.get_full_name()}'
        )
    
    return redirect('admin_lead_detail', lead_id=lead_id)

@login_required
@require_http_methods(["POST"])
def admin_lead_update_status(request, lead_id):
    """Update lead status"""
    user = request.user
    lead = get_object_or_404(Lead, lead_id=lead_id)
    
    # Check permissions: superuser or assigned lead manager
    if not user.is_superuser:
        if not hasattr(user, 'profile') or user.profile.role != 'lead_manager':
            return redirect('staff_login')
        if lead.assigned_to != user:
            return redirect('lead_manager_dashboard')
    
    new_status = request.POST.get('status')
    
    if new_status:
        old_status = lead.get_status_display()
        lead.status = new_status
        
        # If status changed to "converted", create a project automatically
        if new_status == 'converted' and not lead.converted_to_project:
            # Find if there's a client user with this email
            try:
                client_user = User.objects.get(email=lead.email)
            except User.DoesNotExist:
                # Create a basic client user if doesn't exist
                client_user = User.objects.create_user(
                    username=lead.email.split('@')[0],
                    email=lead.email,
                    first_name=lead.first_name,
                    last_name=lead.last_name
                )
                # Create client profile
                UserProfile.objects.create(
                    user=client_user,
                    role='client'
                )
            
            # Create project from lead
            project = Project.objects.create(
                title=f"Project for {lead.get_full_name()}",
                description=f"Converted from {lead.get_source_display()} lead",
                client=client_user,
                lead=lead,
                status='pending_payment',
                lead_manager=lead.assigned_to
            )
            
            # Link project to lead
            lead.converted_to_project = project
            lead.converted_at = timezone.now()
            
            # Create notification for lead manager
            if lead.assigned_to:
                Notification.objects.create(
                    user=lead.assigned_to,
                    title='Lead Converted to Project',
                    message=f'Lead {lead.get_full_name()} has been converted to a project. Please assign it to CRM staff.',
                    type='project_assigned',
                    link=f'/crm-admin/leads/{lead.lead_id}/'
                )
        
        lead.save()
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action='Lead Status Updated',
            model_name='Lead',
            object_id=str(lead.lead_id),
            description=f'Lead status changed from {old_status} to {lead.get_status_display()}'
        )
    
    return redirect('admin_lead_detail', lead_id=lead_id)

@login_required
@require_http_methods(["POST"])
def admin_lead_update_notes(request, lead_id):
    """Update lead notes"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    lead = get_object_or_404(Lead, lead_id=lead_id)
    notes = request.POST.get('notes', '')
    
    lead.notes = notes
    lead.save()
    
    # Log activity
    ActivityLog.objects.create(
        user=user,
        action='Lead Notes Updated',
        model_name='Lead',
        object_id=str(lead.lead_id),
        description=f'Notes updated for lead: {lead.get_full_name()}'
    )
    
    return redirect('admin_lead_detail', lead_id=lead_id)

@login_required
def admin_lead_convert(request, lead_id):
    """Convert lead to project WITHOUT creating user account"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    lead = get_object_or_404(Lead, lead_id=lead_id)
    
    if request.method == 'POST':
        try:
            # Check if client account already exists with this email
            client_user = User.objects.filter(email=lead.email).first()
            
            # Create project from lead
            project_title = request.POST.get('project_title', f"Project for {lead.get_full_name()}")
            project_description = request.POST.get('project_description', f"Converted from lead submission")
            
            # Create project - with or without user account
            project = Project.objects.create(
                title=project_title,
                description=project_description,
                client=client_user,  # Will be None if user doesn't exist yet
                client_email=lead.email,  # Store email for later linking
                lead=lead,
                status='pending_payment'
            )
            
            # Update lead status
            lead.status = 'converted'
            lead.converted_to_project = project
            lead.converted_at = timezone.now()
            lead.save()
            
            # If client account exists, send notification
            if client_user:
                Notification.objects.create(
                    user=client_user,
                    title='New Project Created',
                    message=f'A new project has been created for you: {project.title}. Please submit payment to activate.',
                    type='project_assigned',
                    link='/client/payments/'
                )
            
            # Log activity
            ActivityLog.objects.create(
                user=user,
                action='Lead Converted',
                model_name='Lead',
                object_id=str(lead.lead_id),
                description=f'Lead converted to project: {project.title} (Email: {lead.email})'
            )
            
            # Redirect to lead detail with success message
            return redirect('admin_lead_detail', lead_id=lead_id)
            
        except Exception as e:
            # Handle errors
            print(f"Error converting lead: {str(e)}")
            return redirect('admin_lead_detail', lead_id=lead_id)
    
    # For GET request, redirect back to lead detail
    return redirect('admin_lead_detail', lead_id=lead_id)

@login_required
def admin_users(request):
    """Admin users management page"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    # Get all users with profiles, excluding superusers/admins
    users = User.objects.filter(is_superuser=False).exclude(profile__role='admin').order_by('-date_joined')
    
    # Get pending staff approvals
    pending_approvals = User.objects.filter(
        is_superuser=False,
        profile__is_approved=False,
        profile__role__in=['crm', 'lead_manager', 'project_manager', 'employee']
    ).order_by('-date_joined')
    
    # Apply filters
    role_filter = request.GET.get('role')
    
    if role_filter:
        users = users.filter(profile__role=role_filter)
    
    context = {
        'user': user,
        'users': users,
        'pending_approvals': pending_approvals
    }
    
    return render(request, 'crm/admin/users.html', context)

@login_required
def admin_user_add(request):
    """Admin add user page"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    if request.method == 'POST':
        try:
            # Get form data
            first_name = request.POST.get('first_name')
            last_name = request.POST.get('last_name')
            email = request.POST.get('email')
            username = request.POST.get('username')
            password = request.POST.get('password')
            password_confirm = request.POST.get('password_confirm')
            role = request.POST.get('role')
            phone = request.POST.get('phone', '')
            
            # Validation
            if password != password_confirm:
                context = {
                    'user': user,
                    'error': 'Passwords do not match.'
                }
                return render(request, 'crm/admin/user_add.html', context)
            
            if User.objects.filter(username=username).exists():
                context = {
                    'user': user,
                    'error': 'Username already exists.'
                }
                return render(request, 'crm/admin/user_add.html', context)
            
            if User.objects.filter(email=email).exists():
                context = {
                    'user': user,
                    'error': 'Email already registered.'
                }
                return render(request, 'crm/admin/user_add.html', context)
            
            # Create user
            new_user = User.objects.create_user(
                username=username,
                email=email,
                password=password,
                first_name=first_name,
                last_name=last_name,
                is_staff=True if role != 'client' else False
            )
            
            # Create profile
            profile = UserProfile.objects.create(
                user=new_user,
                role=role,
                created_by=user
            )
            
            if phone:
                profile.set_phone_number(phone)
                profile.save()
            
            # Log activity
            ActivityLog.objects.create(
                user=user,
                action='User Created',
                model_name='User',
                object_id=str(new_user.id),
                description=f'Created new user: {new_user.get_full_name()} ({role})'
            )
            
            # Create notification for new user
            Notification.objects.create(
                user=new_user,
                title='Welcome to Immigration CRM',
                message=f'Your account has been created by {user.get_full_name()}. You can now login with your credentials.',
                type='project_assigned'
            )
            
            context = {
                'user': user,
                'success': f'User {new_user.get_full_name()} created successfully!'
            }
            return render(request, 'crm/admin/user_add.html', context)
            
        except Exception as e:
            context = {
                'user': user,
                'error': f'Error creating user: {str(e)}'
            }
            return render(request, 'crm/admin/user_add.html', context)
    
    context = {
        'user': user
    }
    return render(request, 'crm/admin/user_add.html', context)

@login_required
def admin_user_edit(request, user_id):
    """Admin edit user page"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    edit_user = get_object_or_404(User, id=user_id)
    
    if request.method == 'POST':
        try:
            # Get form data
            first_name = request.POST.get('first_name')
            last_name = request.POST.get('last_name')
            email = request.POST.get('email')
            username = request.POST.get('username')
            password = request.POST.get('password')
            password_confirm = request.POST.get('password_confirm')
            role = request.POST.get('role')
            phone = request.POST.get('phone', '')
            
            # Validation
            if password and password != password_confirm:
                context = {
                    'user': user,
                    'edit_user': edit_user,
                    'error': 'Passwords do not match.'
                }
                return render(request, 'crm/admin/user_edit.html', context)
            
            # Check if username is taken by another user
            if User.objects.filter(username=username).exclude(id=edit_user.id).exists():
                context = {
                    'user': user,
                    'edit_user': edit_user,
                    'error': 'Username already exists.'
                }
                return render(request, 'crm/admin/user_edit.html', context)
            
            # Check if email is taken by another user
            if User.objects.filter(email=email).exclude(id=edit_user.id).exists():
                context = {
                    'user': user,
                    'edit_user': edit_user,
                    'error': 'Email already registered.'
                }
                return render(request, 'crm/admin/user_edit.html', context)
            
            # Update user
            edit_user.first_name = first_name
            edit_user.last_name = last_name
            edit_user.email = email
            edit_user.username = username
            
            # Update password if provided
            if password:
                edit_user.set_password(password)
            
            edit_user.save()
            
            # Update profile
            if hasattr(edit_user, 'profile'):
                profile = edit_user.profile
                profile.role = role
                
                if phone:
                    profile.set_phone_number(phone)
                else:
                    profile.phone_number_encrypted = b''
                
                profile.save()
            
            # Log activity
            ActivityLog.objects.create(
                user=user,
                action='User Updated',
                model_name='User',
                object_id=str(edit_user.id),
                description=f'Updated user: {edit_user.get_full_name()} ({role})'
            )
            
            context = {
                'user': user,
                'edit_user': edit_user,
                'success': f'User {edit_user.get_full_name()} updated successfully!'
            }
            return render(request, 'crm/admin/user_edit.html', context)
            
        except Exception as e:
            context = {
                'user': user,
                'edit_user': edit_user,
                'error': f'Error updating user: {str(e)}'
            }
            return render(request, 'crm/admin/user_edit.html', context)
    
    context = {
        'user': user,
        'edit_user': edit_user
    }
    return render(request, 'crm/admin/user_edit.html', context)

@login_required
@require_http_methods(["POST"])
def admin_user_delete(request, user_id):
    """Admin delete user"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    delete_user = get_object_or_404(User, id=user_id)
    
    # Prevent deleting superusers
    if delete_user.is_superuser:
        return redirect('admin_users')
    
    # Log activity before deletion
    ActivityLog.objects.create(
        user=user,
        action='User Deleted',
        model_name='User',
        object_id=str(delete_user.id),
        description=f'Deleted user: {delete_user.get_full_name()} ({delete_user.username})'
    )
    
    # Delete the user
    delete_user.delete()
    
    return redirect('admin_users')

@login_required
@require_http_methods(["POST"])
def admin_approve_staff(request, user_id):
    """Admin approve staff registration"""
    admin_user = request.user
    
    if not admin_user.is_superuser:
        return redirect('staff_login')
    
    staff_user = get_object_or_404(User, id=user_id)
    profile = staff_user.profile
    
    # Approve the staff member
    profile.is_approved = True
    profile.approved_by = admin_user
    profile.approved_at = timezone.now()
    profile.save()
    
    # Notify the staff member
    Notification.objects.create(
        user=staff_user,
        title='Account Approved',
        message=f'Your account has been approved by {admin_user.get_full_name()}. You can now login to the system.',
        type='project_assigned',
        link='/staff/login/'
    )
    
    # Log activity
    ActivityLog.objects.create(
        user=admin_user,
        action='Staff Approved',
        model_name='UserProfile',
        object_id=str(profile.id),
        description=f'Approved staff registration: {staff_user.get_full_name()} ({profile.get_role_display()})'
    )
    
    return redirect('admin_users')

@login_required
@require_http_methods(["POST"])
def admin_reject_staff(request, user_id):
    """Admin reject staff registration"""
    admin_user = request.user
    
    if not admin_user.is_superuser:
        return redirect('staff_login')
    
    staff_user = get_object_or_404(User, id=user_id)
    profile = staff_user.profile
    
    # Log activity before deletion
    ActivityLog.objects.create(
        user=admin_user,
        action='Staff Rejected',
        model_name='UserProfile',
        object_id=str(profile.id),
        description=f'Rejected staff registration: {staff_user.get_full_name()} ({profile.get_role_display()})'
    )
    
    # Delete the user account
    staff_user.delete()
    
    return redirect('admin_users')

@login_required
@require_http_methods(["POST"])
def admin_verify_client(request, user_id):
    """Admin approve client ID verification"""
    admin_user = request.user
    
    if not admin_user.is_superuser:
        return redirect('staff_login')
    
    client_user = get_object_or_404(User, id=user_id)
    profile = client_user.profile
    
    # Verify the client
    profile.verification_status = 'verified'  # Changed from 'approved' to 'verified'
    profile.verified_by = admin_user
    profile.verified_at = timezone.now()
    profile.verification_notes = ''  # Clear any previous rejection notes
    profile.save()
    
    # Notify the client
    Notification.objects.create(
        user=client_user,
        title='Account Verified',
        message=f'Your ID proof has been verified by our team. Your account is now fully verified.',
        type='project_assigned',
        link='/client/profile/'
    )
    
    # Log activity
    ActivityLog.objects.create(
        user=admin_user,
        action='Client Verified',
        model_name='UserProfile',
        object_id=str(profile.id),
        description=f'Approved ID verification for client: {client_user.get_full_name()}'
    )
    
    return redirect('admin_user_edit', user_id=user_id)

@login_required
@require_http_methods(["POST"])
def admin_reject_verification(request, user_id):
    """Admin reject client ID verification"""
    admin_user = request.user
    
    if not admin_user.is_superuser:
        return redirect('staff_login')
    
    client_user = get_object_or_404(User, id=user_id)
    profile = client_user.profile
    
    # Get rejection notes
    rejection_notes = request.POST.get('rejection_notes', '').strip()
    
    if not rejection_notes:
        rejection_notes = 'ID proof did not meet verification requirements.'
    
    # Reject the verification
    profile.verification_status = 'rejected'
    profile.verified_by = admin_user
    profile.verified_at = timezone.now()
    profile.verification_notes = rejection_notes
    profile.save()
    
    # Notify the client
    Notification.objects.create(
        user=client_user,
        title='Verification Rejected',
        message=f'Your ID proof verification was rejected. Reason: {rejection_notes}. Please upload a new ID proof.',
        type='project_assigned',
        link='/client/profile/'
    )
    
    # Log activity
    ActivityLog.objects.create(
        user=admin_user,
        action='Verification Rejected',
        model_name='UserProfile',
        object_id=str(profile.id),
        description=f'Rejected ID verification for client: {client_user.get_full_name()}. Reason: {rejection_notes}'
    )
    
    return redirect('admin_user_edit', user_id=user_id)

# ============================================
# ADMIN CONTACTS VIEWS
# ============================================

@login_required
def admin_contacts(request):
    """Admin contacts management page"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    from main.models import Contact
    
    # Get all contacts
    contacts = Contact.objects.all().order_by('-submitted_at')
    
    # Apply filters
    read_filter = request.GET.get('read')
    
    if read_filter == 'unread':
        contacts = contacts.filter(is_read=False)
    elif read_filter == 'read':
        contacts = contacts.filter(is_read=True)
    
    context = {
        'user': user,
        'contacts': contacts
    }
    
    return render(request, 'crm/admin/contacts.html', context)

@login_required
def admin_contact_detail(request, contact_id):
    """Admin contact detail page"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    from main.models import Contact
    
    contact = get_object_or_404(Contact, id=contact_id)
    
    # Mark as read when viewed
    if not contact.is_read:
        contact.is_read = True
        contact.save()
    
    context = {
        'user': user,
        'contact': contact
    }
    
    return render(request, 'crm/admin/contact_detail.html', context)

@login_required
@require_http_methods(["POST"])
def admin_contact_toggle_read(request, contact_id):
    """Toggle contact read status"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    from main.models import Contact
    
    contact = get_object_or_404(Contact, id=contact_id)
    contact.is_read = not contact.is_read
    contact.save()
    
    return redirect('admin_contacts')

# ============================================
# CRM DASHBOARD VIEWS
# ============================================

@login_required
def crm_dashboard(request):
    """CRM dashboard"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'crm':
        return redirect('staff_login')
    
    # Get assigned projects
    projects = Project.objects.filter(crm=user).order_by('-created_at')
    
    # Apply status filter if provided
    status_filter = request.GET.get('status')
    if status_filter:
        projects = projects.filter(status=status_filter)
    
    # Get ALL payments for assigned projects (pending, approved, rejected)
    all_payments = Payment.objects.filter(
        project__crm=user
    ).order_by('-created_at')
    
    # Get pending payments for assigned projects
    pending_payments = Payment.objects.filter(
        project__crm=user,
        status='pending'
    ).order_by('-created_at')
    
    # Statistics (always show total counts, not filtered)
    all_projects = Project.objects.filter(crm=user)
    stats = {
        'assigned_projects': all_projects.count(),
        'active_projects': all_projects.filter(status='active').count(),
        'pending_payments': pending_payments.count()
    }
    
    context = {
        'user': user,
        'projects': projects,
        'pending_payments': pending_payments,
        'all_payments': all_payments,
        'stats': stats
    }
    
    return render(request, 'crm/crm/dashboard.html', context)

@login_required
def crm_project_detail(request, project_id):
    """CRM project detail view"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'crm':
        return redirect('staff_login')
    
    project = get_object_or_404(Project, project_id=project_id, crm=user)
    
    # Get all project managers for assignment
    project_managers = User.objects.filter(profile__role='project_manager')
    
    # Get chat messages
    chat_messages = ProjectChat.objects.filter(project=project).order_by('timestamp')
    
    # Get project files
    project_files = ProjectFile.objects.filter(project=project).order_by('-uploaded_at')
    
    # Get task files (files uploaded by employees)
    task_files = TaskFile.objects.filter(task__project=project).select_related('task', 'uploaded_by').order_by('-uploaded_at')
    
    context = {
        'user': user,
        'project': project,
        'project_managers': project_managers,
        'chat_messages': chat_messages,
        'project_files': project_files,
        'task_files': task_files
    }
    
    return render(request, 'crm/crm/project_detail.html', context)

@login_required
@require_http_methods(["POST"])
def crm_assign_pm(request, project_id):
    """CRM assign project manager to project"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'crm':
        return redirect('staff_login')
    
    project = get_object_or_404(Project, project_id=project_id, crm=user)
    project_manager_id = request.POST.get('project_manager_id')
    
    if project_manager_id:
        pm_user = get_object_or_404(User, id=project_manager_id)
        project.project_manager = pm_user
        project.save()
        
        # Create notification for Project Manager
        Notification.objects.create(
            user=pm_user,
            title='New Project Assigned',
            message=f'You have been assigned a new project: {project.title}',
            type='project_assigned',
            link=f'/pm/projects/{project.project_id}/'
        )
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action='Project Manager Assigned',
            model_name='Project',
            object_id=str(project.project_id),
            description=f'Project Manager assigned: {pm_user.get_full_name()}'
        )
    
    return redirect('crm_project_detail', project_id=project_id)

@login_required
@require_http_methods(["POST"])
def crm_send_message(request, project_id):
    """CRM send message to project"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'crm':
        return redirect('staff_login')
    
    project = get_object_or_404(Project, project_id=project_id, crm=user)
    
    message_text = request.POST.get('message', '').strip()
    
    if message_text:
        # Create chat message
        chat_message = ProjectChat()
        chat_message.project = project
        chat_message.sender = user
        chat_message.set_message(message_text)
        chat_message.save()
        
        # Create notifications for all project participants
        participants = []
        
        # Add client
        if project.client:
            participants.append(project.client)
        
        # Add CRM staff (skip if sender)
        if project.crm and project.crm != user:
            participants.append(project.crm)
        
        # Add Project Manager
        if project.project_manager:
            participants.append(project.project_manager)
        
        # Send notification to everyone except the sender
        for participant in participants:
            if participant != user:
                # Determine the correct link based on role
                if hasattr(participant, 'profile'):
                    if participant.profile.role == 'client':
                        link = f'/client/projects/{project.project_id}/'
                    elif participant.profile.role == 'project_manager':
                        link = f'/pm/projects/{project.project_id}/'
                    elif participant.profile.role == 'crm':
                        link = f'/crm/projects/{project.project_id}/'
                    else:
                        link = f'/client/projects/{project.project_id}/'
                else:
                    link = f'/client/projects/{project.project_id}/'
                
                Notification.objects.create(
                    user=participant,
                    title='New Project Message',
                    message=f'{user.get_full_name()} sent a message in {project.title}',
                    type='message',
                    link=link
                )
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action='Chat Message Sent',
            model_name='ProjectChat',
            object_id=str(chat_message.message_id),
            description=f'CRM sent message in project: {project.title}'
        )
    
    return redirect('crm_project_detail', project_id=project_id)

@login_required
@require_http_methods(["POST"])
def crm_upload_file(request, project_id):
    """CRM upload project file"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'crm':
        return redirect('staff_login')
    
    project = get_object_or_404(Project, project_id=project_id, crm=user)
    
    uploaded_file = request.FILES.get('file')
    
    if uploaded_file:
        # Create project file
        project_file = ProjectFile.objects.create(
            project=project,
            file=uploaded_file,
            uploaded_by=user
        )
        
        # Notify client
        if project.client:
            Notification.objects.create(
                user=project.client,
                title='New File Uploaded',
                message=f'CRM staff uploaded a new file to your project: {project.title}',
                type='file_uploaded',
                link=f'/client/projects/{project.project_id}/'
            )
        
        # Notify PM
        if project.project_manager:
            Notification.objects.create(
                user=project.project_manager,
                title='New File Uploaded',
                message=f'CRM staff uploaded a new file to project: {project.title}',
                type='file_uploaded',
                link=f'/pm/projects/{project.project_id}/'
            )
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action='File Uploaded',
            model_name='ProjectFile',
            object_id=str(project_file.id),
            description=f'CRM uploaded file to project: {project.title}'
        )
    
    return redirect('crm_project_detail', project_id=project_id)

@login_required
@require_http_methods(["POST"])
def crm_update_payment_status(request, payment_id):
    """CRM update payment status"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'crm':
        return redirect('staff_login')
    
    payment = get_object_or_404(Payment, payment_id=payment_id, project__crm=user)
    new_status = request.POST.get('status')
    notes = request.POST.get('notes', '')
    
    if new_status in ['approved', 'rejected']:
        payment.status = new_status
        payment.approved_by = user
        payment.approved_at = timezone.now()
        payment.notes = notes
        payment.save()
        
        # Update project status if payment approved
        if new_status == 'approved':
            payment.project.status = 'active'
            payment.project.save()
            
            # Generate invoice
            invoice_number = payment.generate_invoice_number()
            
            # Notify client with invoice
            Notification.objects.create(
                user=payment.client,
                title='Payment Approved - Invoice Generated',
                message=f'Your payment for {payment.project.title} has been approved. Invoice #{invoice_number} has been generated.',
                type='payment_approved',
                link=f'/client/payments/'
            )
        else:
            # Notify client of rejection
            Notification.objects.create(
                user=payment.client,
                title='Payment Rejected',
                message=f'Your payment for {payment.project.title} was rejected. Please resubmit.',
                type='payment_rejected',
                link=f'/client/payments/'
            )
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action=f'Payment {new_status.title()}',
            model_name='Payment',
            object_id=str(payment.payment_id),
            description=f'Payment {new_status} for project: {payment.project.title}'
        )
    
    return redirect('crm_dashboard')

@login_required
@require_http_methods(["POST"])
def admin_update_payment_status(request, payment_id):
    """Admin update payment status"""
    user = request.user
    
    if not user.is_superuser:
        return redirect('staff_login')
    
    payment = get_object_or_404(Payment, payment_id=payment_id)
    new_status = request.POST.get('status')
    notes = request.POST.get('notes', '')
    
    if new_status in ['approved', 'rejected']:
        payment.status = new_status
        payment.approved_by = user
        payment.approved_at = timezone.now()
        payment.notes = notes
        payment.save()
        
        # Update project status if payment approved
        if new_status == 'approved':
            payment.project.status = 'active'
            payment.project.save()
            
            # Generate invoice
            invoice_number = payment.generate_invoice_number()
            
            # Notify client with invoice
            Notification.objects.create(
                user=payment.client,
                title='Payment Approved - Invoice Generated',
                message=f'Your payment for {payment.project.title} has been approved. Invoice #{invoice_number} has been generated.',
                type='payment_approved',
                link=f'/client/payments/'
            )
        else:
            # Notify client of rejection
            Notification.objects.create(
                user=payment.client,
                title='Payment Rejected',
                message=f'Your payment for {payment.project.title} was rejected. Please resubmit.',
                type='payment_rejected',
                link=f'/client/payments/'
            )
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action=f'Payment {new_status.title()}',
            model_name='Payment',
            object_id=str(payment.payment_id),
            description=f'Payment {new_status} for project: {payment.project.title}'
        )
    
    return redirect('admin_payments')

# ============================================
# LEAD MANAGER DASHBOARD VIEWS
# ============================================

@login_required
def lead_manager_dashboard(request):
    """Lead Manager dashboard"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'lead_manager':
        return redirect('staff_login')
    
    # Get assigned leads
    leads = Lead.objects.filter(assigned_to=user).order_by('-created_at')
    
    # Statistics
    stats = {
        'assigned_leads': leads.count(),
        'new_leads': leads.filter(status='new').count(),
        'contacted_leads': leads.filter(status='contacted').count(),
        'qualified_leads': leads.filter(status='qualified').count(),
        'converted_leads': leads.filter(status='converted').count()
    }
    
    context = {
        'user': user,
        'leads': leads,
        'stats': stats
    }
    
    return render(request, 'crm/lead_manager/dashboard.html', context)

@login_required
def lead_manager_view_assessment(request, lead_id):
    """Lead Manager view assessment details"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'lead_manager':
        return redirect('staff_login')
    
    lead = get_object_or_404(Lead, lead_id=lead_id, assigned_to=user)
    
    if not lead.assessment:
        return redirect('admin_lead_detail', lead_id=lead_id)
    
    context = {
        'user': user,
        'lead': lead,
        'assessment': lead.assessment
    }
    
    return render(request, 'crm/lead_manager/assessment_view.html', context)

# ============================================
# PROJECT MANAGER DASHBOARD VIEWS
# ============================================

@login_required
def pm_dashboard(request):
    """Project Manager dashboard"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'project_manager':
        return redirect('staff_login')
    
    # Get assigned projects
    projects = Project.objects.filter(project_manager=user).order_by('-created_at')
    
    # Get all tasks
    tasks = Task.objects.filter(project__project_manager=user).order_by('-created_at')
    
    # Statistics
    stats = {
        'assigned_projects': projects.count(),
        'total_tasks': tasks.count(),
        'completed_tasks': tasks.filter(status='completed').count(),
        'pending_tasks': tasks.filter(status='pending').count()
    }
    
    context = {
        'user': user,
        'projects': projects,
        'tasks': tasks[:10],
        'stats': stats
    }
    
    return render(request, 'crm/project_manager/dashboard.html', context)

@login_required
def pm_project_detail(request, project_id):
    """Project Manager project detail view"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'project_manager':
        return redirect('staff_login')
    
    project = get_object_or_404(Project, project_id=project_id, project_manager=user)
    
    # Get tasks for this project
    tasks = Task.objects.filter(project=project).order_by('-created_at')
    
    # Calculate task completion statistics
    total_tasks = tasks.count()
    completed_tasks = tasks.filter(status='completed').count()
    in_progress_tasks = tasks.filter(status='in_progress').count()
    pending_tasks = tasks.filter(status='pending').count()
    
    # Calculate completion percentage
    completion_percentage = 0
    if total_tasks > 0:
        completion_percentage = round((completed_tasks / total_tasks) * 100)
    
    # Get all employees for task assignment
    employees = User.objects.filter(profile__role='employee')
    
    # Get chat messages
    chat_messages = ProjectChat.objects.filter(project=project).order_by('timestamp')
    
    # Get project files
    project_files = ProjectFile.objects.filter(project=project).order_by('-uploaded_at')
    
    # Get all task files from this project's tasks
    task_files = TaskFile.objects.filter(task__project=project).select_related('task', 'uploaded_by').order_by('-uploaded_at')
    
    context = {
        'user': user,
        'project': project,
        'tasks': tasks,
        'employees': employees,
        'total_tasks': total_tasks,
        'completed_tasks': completed_tasks,
        'in_progress_tasks': in_progress_tasks,
        'pending_tasks': pending_tasks,
        'completion_percentage': completion_percentage,
        'chat_messages': chat_messages,
        'project_files': project_files,
        'task_files': task_files
    }
    
    return render(request, 'crm/project_manager/project_detail.html', context)

@login_required
@require_http_methods(["POST"])
def pm_create_task(request, project_id):
    """Project Manager create task"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'project_manager':
        return redirect('staff_login')
    
    project = get_object_or_404(Project, project_id=project_id, project_manager=user)
    
    try:
        title = request.POST.get('title')
        description = request.POST.get('description')
        priority = request.POST.get('priority', 'medium')
        due_date = request.POST.get('due_date')
        assigned_to_id = request.POST.get('assigned_to')
        
        # Create task
        task = Task.objects.create(
            project=project,
            title=title,
            description=description,
            priority=priority,
            due_date=due_date if due_date else None,
            assigned_by=user
        )
        
        # Assign to employee if selected
        if assigned_to_id:
            assigned_user = get_object_or_404(User, id=assigned_to_id)
            task.assigned_to = assigned_user
            task.save()
            
            # Create notification for assigned employee
            Notification.objects.create(
                user=assigned_user,
                title='New Task Assigned',
                message=f'You have been assigned a new task: {task.title}',
                type='project_assigned',
                link=f'/employee/tasks/{task.task_id}/'
            )
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action='Task Created',
            model_name='Task',
            object_id=str(task.task_id),
            description=f'Created task: {task.title} for project: {project.title}'
        )
        
    except Exception as e:
        print(f"Error creating task: {e}")
    
    return redirect('pm_project_detail', project_id=project_id)

@login_required
@require_http_methods(["POST"])
def pm_send_message(request, project_id):
    """Project Manager send chat message"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'project_manager':
        return redirect('staff_login')
    
    project = get_object_or_404(Project, project_id=project_id, project_manager=user)
    
    message_text = request.POST.get('message', '').strip()
    
    if message_text:
        # Create chat message
        chat_message = ProjectChat()
        chat_message.project = project
        chat_message.sender = user
        chat_message.set_message(message_text)
        chat_message.save()
        
        # Create notifications for all project participants
        participants = []
        
        # Add client
        if project.client:
            participants.append(project.client)
        
        # Add CRM staff
        if project.crm:
            participants.append(project.crm)
        
        # Add Project Manager
        if project.project_manager:
            participants.append(project.project_manager)
        
        # Send notification to everyone except the sender
        for participant in participants:
            if participant != user:
                # Determine the correct link based on role
                if hasattr(participant, 'profile'):
                    if participant.profile.role == 'client':
                        link = f'/client/projects/{project.project_id}/'
                    elif participant.profile.role == 'project_manager':
                        link = f'/pm/projects/{project.project_id}/'
                    elif participant.profile.role == 'crm':
                        link = f'/crm/projects/{project.project_id}/'
                    else:
                        link = f'/client/projects/{project.project_id}/'
                else:
                    link = f'/client/projects/{project.project_id}/'
                
                Notification.objects.create(
                    user=participant,
                    title='New Project Message',
                    message=f'{user.get_full_name()} sent a message in {project.title}',
                    type='message',
                    link=link
                )
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action='Chat Message Sent',
            model_name='ProjectChat',
            object_id=str(chat_message.message_id),
            description=f'Sent message in project: {project.title}'
        )
    
    return redirect('pm_project_detail', project_id=project_id)

@login_required
@require_http_methods(["POST"])
def pm_complete_project(request, project_id):
    """Project Manager mark project as complete"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'project_manager':
        return redirect('staff_login')
    
    project = get_object_or_404(Project, project_id=project_id, project_manager=user)
    
    # Update project status to completed
    old_status = project.get_status_display()
    project.status = 'completed'
    project.save()
    
    # Notify client
    if project.client:
        Notification.objects.create(
            user=project.client,
            title='Project Completed',
            message=f'Your project "{project.title}" has been completed!',
            type='project_assigned',
            link=f'/crm/client/projects/'
        )
    
    # Notify CRM staff
    if project.crm:
        Notification.objects.create(
            user=project.crm,
            title='Project Completed',
            message=f'Project "{project.title}" has been marked as complete by {user.get_full_name()}',
            type='project_assigned',
            link=f'/crm/projects/{project.project_id}/'
        )
    
    # Notify Lead Manager
    if project.lead_manager:
        Notification.objects.create(
            user=project.lead_manager,
            title='Project Completed',
            message=f'Project "{project.title}" has been completed',
            type='project_assigned',
            link=f'/crm-admin/projects/'
        )
    
    # Log activity
    ActivityLog.objects.create(
        user=user,
        action='Project Completed',
        model_name='Project',
        object_id=str(project.project_id),
        description=f'Project status changed from {old_status} to Completed'
    )
    
    return redirect('pm_project_detail', project_id=project_id)

@login_required
@require_http_methods(["POST"])
def pm_upload_project_file(request, project_id):
    """Project Manager upload project file"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'project_manager':
        return redirect('staff_login')
    
    project = get_object_or_404(Project, project_id=project_id, project_manager=user)
    
    try:
        file = request.FILES.get('file')
        description = request.POST.get('description', '').strip()
        
        if file and description:
            # Check file size (max 50MB)
            if file.size > 50 * 1024 * 1024:
                return redirect('pm_project_detail', project_id=project_id)
            
            # Check file extension
            allowed_extensions = ['.pdf', '.doc', '.docx', '.zip', '.rar']
            file_ext = file.name[file.name.rfind('.'):].lower()
            
            if file_ext not in allowed_extensions:
                return redirect('pm_project_detail', project_id=project_id)
            
            # Create project file
            project_file = ProjectFile.objects.create(
                project=project,
                file=file,
                description=description,
                uploaded_by=user
            )
            
            # Notify client
            if project.client:
                Notification.objects.create(
                    user=project.client,
                    title='New Project File',
                    message=f'A new file "{description}" has been uploaded to your project',
                    type='project_assigned',
                    link=f'/crm/client/projects/'
                )
            
            # Notify CRM staff
            if project.crm:
                Notification.objects.create(
                    user=project.crm,
                    title='New Project File',
                    message=f'PM uploaded "{description}" to project: {project.title}',
                    type='project_assigned',
                    link=f'/crm/projects/{project.project_id}/'
                )
            
            # Notify Lead Manager
            if project.lead_manager:
                Notification.objects.create(
                    user=project.lead_manager,
                    title='New Project File',
                    message=f'New file uploaded to project: {project.title}',
                    type='project_assigned',
                    link=f'/crm-admin/projects/'
                )
            
            # Log activity
            ActivityLog.objects.create(
                user=user,
                action='Project File Uploaded',
                model_name='ProjectFile',
                object_id=str(project_file.id),
                description=f'Uploaded "{description}" to project: {project.title}'
            )
    
    except Exception as e:
        print(f"Error uploading file: {e}")
    
    return redirect('pm_project_detail', project_id=project_id)

@login_required
@require_http_methods(["POST"])
def pm_delete_project_file(request, file_id):
    """Project Manager delete project file"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'project_manager':
        return redirect('staff_login')
    
    project_file = get_object_or_404(ProjectFile, id=file_id, uploaded_by=user)
    project_id = project_file.project.project_id
    
    # Delete the file
    project_file.file.delete()
    project_file.delete()
    
    # Log activity
    ActivityLog.objects.create(
        user=user,
        action='Project File Deleted',
        model_name='ProjectFile',
        object_id=str(file_id),
        description=f'Deleted file from project: {project_file.project.title}'
    )
    
    return redirect('pm_project_detail', project_id=project_id)

# ============================================
# EMPLOYEE DASHBOARD VIEWS
# ============================================

@login_required
def employee_dashboard(request):
    """Employee dashboard"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'employee':
        return redirect('staff_login')
    
    # Get assigned tasks
    tasks = Task.objects.filter(assigned_to=user).order_by('-created_at')
    
    # Statistics
    stats = {
        'assigned_tasks': tasks.count(),
        'completed_tasks': tasks.filter(status='completed').count(),
        'in_progress_tasks': tasks.filter(status='in_progress').count(),
        'pending_tasks': tasks.filter(status='pending').count()
    }
    
    context = {
        'user': user,
        'tasks': tasks,
        'stats': stats
    }
    
    return render(request, 'crm/employee/dashboard.html', context)

@login_required
@require_http_methods(["POST"])
def employee_update_task_status(request, task_id):
    """Employee update task status"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'employee':
        return redirect('staff_login')
    
    task = get_object_or_404(Task, task_id=task_id, assigned_to=user)
    new_status = request.POST.get('status')
    
    if new_status in ['pending', 'in_progress', 'completed']:
        old_status = task.get_status_display()
        task.status = new_status
        
        # Set completed_at timestamp if marking as completed
        if new_status == 'completed':
            task.completed_at = timezone.now()
        
        task.save()
        
        # Notify Project Manager
        if task.project.project_manager:
            Notification.objects.create(
                user=task.project.project_manager,
                title='Task Status Updated',
                message=f'{user.get_full_name()} updated task "{task.title}" to {task.get_status_display()}',
                type='project_assigned',
                link=f'/pm/projects/{task.project.project_id}/'
            )
        
        # Log activity
        ActivityLog.objects.create(
            user=user,
            action='Task Status Updated',
            model_name='Task',
            object_id=str(task.task_id),
            description=f'Task status changed from {old_status} to {task.get_status_display()}'
        )
    
    return redirect('employee_dashboard')

@login_required
def employee_task_detail(request, task_id):
    """Employee task detail view"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'employee':
        return redirect('staff_login')
    
    task = get_object_or_404(Task, task_id=task_id, assigned_to=user)
    
    context = {
        'user': user,
        'task': task
    }
    
    return render(request, 'crm/employee/task_detail.html', context)

@login_required
@require_http_methods(["POST"])
def employee_upload_file(request, task_id):
    """Employee upload file to task"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'employee':
        return redirect('staff_login')
    
    task = get_object_or_404(Task, task_id=task_id, assigned_to=user)
    
    try:
        file = request.FILES.get('file')
        description = request.POST.get('description', '')
        
        if file:
            # Check file size (max 10MB)
            if file.size > 10 * 1024 * 1024:
                # Redirect with error (you can add messages framework later)
                return redirect('employee_task_detail', task_id=task_id)
            
            # Check file extension
            allowed_extensions = ['.pdf', '.doc', '.docx']
            file_ext = file.name[file.name.rfind('.'):].lower()
            
            if file_ext not in allowed_extensions:
                return redirect('employee_task_detail', task_id=task_id)
            
            # Create task file
            task_file = TaskFile.objects.create(
                task=task,
                file=file,
                description=description,
                uploaded_by=user
            )
            
            # Notify Project Manager
            if task.project.project_manager:
                Notification.objects.create(
                    user=task.project.project_manager,
                    title='New File Uploaded',
                    message=f'{user.get_full_name()} uploaded a file to task "{task.title}"',
                    type='project_assigned',
                    link=f'/pm/projects/{task.project.project_id}/'
                )
            
            # Log activity
            ActivityLog.objects.create(
                user=user,
                action='File Uploaded',
                model_name='TaskFile',
                object_id=str(task_file.file_id),
                description=f'Uploaded file "{file.name}" to task: {task.title}'
            )
    
    except Exception as e:
        print(f"Error uploading file: {e}")
    
    return redirect('employee_task_detail', task_id=task_id)

@login_required
@require_http_methods(["POST"])
def employee_delete_file(request, file_id):
    """Employee delete uploaded file"""
    user = request.user
    
    if not hasattr(user, 'profile') or user.profile.role != 'employee':
        return redirect('staff_login')
    
    task_file = get_object_or_404(TaskFile, file_id=file_id, uploaded_by=user)
    task_id = task_file.task.task_id
    
    # Delete the file
    task_file.file.delete()
    task_file.delete()
    
    # Log activity
    ActivityLog.objects.create(
        user=user,
        action='File Deleted',
        model_name='TaskFile',
        object_id=str(file_id),
        description=f'Deleted file from task: {task_file.task.title}'
    )
    
    return redirect('employee_task_detail', task_id=task_id)

# ============================================
# GENERIC STAFF DASHBOARD (FALLBACK)
# ============================================

@login_required
def staff_dashboard(request):
    """Generic staff dashboard"""
    user = request.user
    
    context = {
        'user': user
    }
    
    return render(request, 'crm/staff/dashboard.html', context)


# ============================================
# NOTIFICATION API ENDPOINTS
# ============================================

@login_required
def get_notifications(request):
    """API endpoint to get user notifications"""
    user = request.user
    
    # Get all recent notifications (both read and unread)
    all_notifications = Notification.objects.filter(
        user=user
    ).order_by('-created_at')[:20]
    
    # Get unread count
    unread_count = Notification.objects.filter(
        user=user,
        is_read=False
    ).count()
    
    # Format notifications for JSON response
    notifications_data = []
    for notif in all_notifications:
        notifications_data.append({
            'id': str(notif.notification_id),
            'title': notif.title,
            'message': notif.message,
            'type': notif.type,
            'link': notif.link,
            'is_read': notif.is_read,
            'created_at': notif.created_at.strftime('%Y-%m-%d %H:%M:%S'),
            'time_ago': get_time_ago(notif.created_at)
        })
    
    return JsonResponse({
        'notifications': notifications_data,
        'count': unread_count
    })

@login_required
@require_http_methods(["POST"])
def mark_notification_read(request, notification_id):
    """Mark a notification as read"""
    user = request.user
    
    try:
        notification = Notification.objects.get(
            notification_id=notification_id,
            user=user
        )
        notification.is_read = True
        notification.save()
        
        return JsonResponse({'success': True})
    except Notification.DoesNotExist:
        return JsonResponse({'success': False, 'error': 'Notification not found'}, status=404)

@login_required
@require_http_methods(["POST"])
def mark_all_notifications_read(request):
    """Mark all notifications as read"""
    user = request.user
    
    Notification.objects.filter(user=user, is_read=False).update(is_read=True)
    
    return JsonResponse({'success': True})

@login_required
def get_project_messages(request, project_id):
    """API endpoint to get project chat messages"""
    user = request.user
    
    # Get project and verify access
    try:
        if hasattr(user, 'profile'):
            if user.profile.role == 'client':
                project = Project.objects.get(project_id=project_id, client=user)
            elif user.profile.role == 'project_manager':
                project = Project.objects.get(project_id=project_id, project_manager=user)
            elif user.profile.role == 'crm':
                project = Project.objects.get(project_id=project_id, crm=user)
            else:
                return JsonResponse({'error': 'Access denied'}, status=403)
        else:
            return JsonResponse({'error': 'Access denied'}, status=403)
    except Project.DoesNotExist:
        return JsonResponse({'error': 'Project not found'}, status=404)
    
    # Get messages
    messages = ProjectChat.objects.filter(project=project).order_by('timestamp')
    
    # Format messages for JSON response
    messages_data = []
    for msg in messages:
        try:
            messages_data.append({
                'id': str(msg.message_id),
                'sender_name': msg.sender.get_full_name(),
                'sender_id': msg.sender.id,
                'message': msg.message,
                'timestamp': msg.timestamp.strftime('%Y-%m-%d %H:%M:%S'),
                'is_own': msg.sender.id == user.id
            })
        except Exception as e:
            # Skip messages that can't be decrypted
            continue
    
    return JsonResponse({
        'messages': messages_data,
        'count': len(messages_data)
    })
    return JsonResponse({'success': True})

def get_time_ago(dt):
    """Helper function to get human-readable time ago"""
    from django.utils import timezone
    from datetime import timedelta
    
    now = timezone.now()
    diff = now - dt
    
    if diff < timedelta(minutes=1):
        return 'Just now'
    elif diff < timedelta(hours=1):
        minutes = int(diff.total_seconds() / 60)
        return f'{minutes} minute{"s" if minutes != 1 else ""} ago'
    elif diff < timedelta(days=1):
        hours = int(diff.total_seconds() / 3600)
        return f'{hours} hour{"s" if hours != 1 else ""} ago'
    elif diff < timedelta(days=7):
        days = diff.days
        return f'{days} day{"s" if days != 1 else ""} ago'
    elif diff < timedelta(days=30):
        weeks = int(diff.days / 7)
        return f'{weeks} week{"s" if weeks != 1 else ""} ago'
    else:
        months = int(diff.days / 30)
        return f'{months} month{"s" if months != 1 else ""} ago'
