"""
Final comprehensive check of assessment encryption
"""

import os
import django

# Setup Django
os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'myproject.settings')
django.setup()

from assessment.models import ProfileAssessment

print("=" * 70)
print("FINAL ENCRYPTION VERIFICATION")
print("=" * 70)

assessments = ProfileAssessment.objects.all().order_by('-created_at')
print(f"\nTotal assessments in database: {assessments.count()}")

if assessments.exists():
    print("\nDetailed Analysis:")
    print("-" * 50)
    
    for i, assessment in enumerate(assessments, 1):
        print(f"\n{i}. Assessment ID: {assessment.assessment_id}")
        print(f"   Database ID: {assessment.id}")
        print(f"   Created: {assessment.created_at}")
        
        # Check encryption
        encrypted_data = {
            'first_name': assessment.first_name_encrypted,
            'last_name': assessment.last_name_encrypted,
            'email': assessment.email_encrypted,
            'phone': assessment.phone_number_encrypted,
            'location': assessment.current_location_encrypted
        }
        
        print(f"   ENCRYPTION STATUS:")
        for field, data in encrypted_data.items():
            if data and len(data) > 0:
                is_encrypted = data.startswith(b'gAAAAAB')
                size = len(data)
                status = "✅ ENCRYPTED" if is_encrypted else "❌ NOT ENCRYPTED"
                print(f"     {field}: {status} ({size} bytes)")
                if is_encrypted:
                    print(f"       Sample: {data[:30]}...")
            else:
                print(f"     {field}: ❌ EMPTY")
        
        print(f"   DECRYPTED DATA:")
        print(f"     Name: {assessment.get_full_name()}")
        print(f"     Email: {assessment.get_email()}")
        print(f"     Phone: {assessment.get_phone_number()}")
        print(f"     Location: {assessment.get_current_location()}")
        
        print(f"   ASSESSMENT RESULTS:")
        print(f"     Score: {assessment.total_score}/30 ({assessment.percentage_score:.1f}%)")
        print(f"     Status: {assessment.eligibility_status}")

print("\n" + "=" * 70)
print("SUMMARY:")
print("=" * 70)

if assessments.exists():
    latest = assessments.first()
    all_encrypted = all([
        latest.first_name_encrypted.startswith(b'gAAAAAB'),
        latest.last_name_encrypted.startswith(b'gAAAAAB'),
        latest.email_encrypted.startswith(b'gAAAAAB'),
        latest.current_location_encrypted.startswith(b'gAAAAAB')
    ])
    
    if all_encrypted:
        print("✅ ALL DATA IS PROPERLY ENCRYPTED!")
        print("✅ Fernet encryption format detected")
        print("✅ Data is secure in database")
        print("✅ Decryption works correctly")
    else:
        print("❌ SOME DATA IS NOT ENCRYPTED!")
        print("❌ Check encryption implementation")
else:
    print("❌ No assessments found in database")

print("\nTo view encrypted data in admin:")
print("Visit: http://127.0.0.1:8000/admin/assessment/profileassessment/")
print("\n" + "=" * 70)