from django.db import models
from cryptography.fernet import Fernet
from django.conf import settings
import base64
import hashlib

class Contact(models.Model):
    """Model to store encrypted contact form submissions"""
    
    # Encrypted fields
    first_name_encrypted = models.BinaryField()
    last_name_encrypted = models.BinaryField()
    email_encrypted = models.BinaryField()
    phone_encrypted = models.BinaryField(null=True, blank=True)
    subject_encrypted = models.BinaryField()
    message_encrypted = models.BinaryField()
    
    # Non-encrypted metadata
    submitted_at = models.DateTimeField(auto_now_add=True)
    ip_address = models.GenericIPAddressField(null=True, blank=True)
    is_read = models.BooleanField(default=False)
    
    class Meta:
        ordering = ['-submitted_at']
        verbose_name = 'Contact Submission'
        verbose_name_plural = 'Contact Submissions'
    
    def __str__(self):
        return f"Contact from {self.get_email()} - {self.submitted_at.strftime('%Y-%m-%d %H:%M')}"
    
    @staticmethod
    def get_cipher():
        """Get encryption cipher"""
        # Generate a key from SECRET_KEY
        key = hashlib.sha256(settings.SECRET_KEY.encode()).digest()
        key_base64 = base64.urlsafe_b64encode(key)
        return Fernet(key_base64)
    
    def encrypt_field(self, value):
        """Encrypt a field value"""
        if not value:
            return b''
        cipher = self.get_cipher()
        return cipher.encrypt(value.encode())
    
    def decrypt_field(self, encrypted_value):
        """Decrypt a field value"""
        if not encrypted_value:
            return ''
        cipher = self.get_cipher()
        return cipher.decrypt(encrypted_value).decode()
    
    # Setter methods
    def set_first_name(self, value):
        self.first_name_encrypted = self.encrypt_field(value)
    
    def set_last_name(self, value):
        self.last_name_encrypted = self.encrypt_field(value)
    
    def set_email(self, value):
        self.email_encrypted = self.encrypt_field(value)
    
    def set_phone(self, value):
        self.phone_encrypted = self.encrypt_field(value) if value else b''
    
    def set_subject(self, value):
        self.subject_encrypted = self.encrypt_field(value)
    
    def set_message(self, value):
        self.message_encrypted = self.encrypt_field(value)
    
    # Getter methods
    def get_first_name(self):
        return self.decrypt_field(self.first_name_encrypted)
    
    def get_last_name(self):
        return self.decrypt_field(self.last_name_encrypted)
    
    def get_email(self):
        return self.decrypt_field(self.email_encrypted)
    
    def get_phone(self):
        return self.decrypt_field(self.phone_encrypted)
    
    def get_subject(self):
        return self.decrypt_field(self.subject_encrypted)
    
    def get_message(self):
        return self.decrypt_field(self.message_encrypted)
    
    def get_full_name(self):
        return f"{self.get_first_name()} {self.get_last_name()}"
