"""
Script to show RAW database content vs decrypted content
This proves data is encrypted in database but decrypted for display
"""

import sqlite3
import os
import django

# Setup Django
os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'myproject.settings')
django.setup()

from assessment.models import ProfileAssessment

print("=" * 80)
print("RAW DATABASE CONTENT vs DECRYPTED CONTENT")
print("=" * 80)

# Connect directly to SQLite database
conn = sqlite3.connect('db.sqlite3')
cursor = conn.cursor()

# Get raw data from database
cursor.execute("""
    SELECT id, first_name_encrypted, last_name_encrypted, email_encrypted, 
           field_of_expertise, total_score, eligibility_status
    FROM assessment_profileassessment
    ORDER BY created_at DESC
    LIMIT 4
""")

rows = cursor.fetchall()

print(f"\nFound {len(rows)} assessments in database\n")

for row in rows:
    db_id, first_encrypted, last_encrypted, email_encrypted, expertise, score, status = row
    
    print(f"Assessment ID: {db_id}")
    print("-" * 50)
    
    print("\n1. RAW DATABASE CONTENT (ENCRYPTED):")
    print(f"   First Name: {first_encrypted[:60]}...")
    print(f"   Last Name:  {last_encrypted[:60]}...")
    print(f"   Email:      {email_encrypted[:60]}...")
    
    # Get the same record through Django ORM (which decrypts)
    assessment = ProfileAssessment.objects.get(id=db_id)
    
    print("\n2. DJANGO ADMIN DISPLAY (DECRYPTED):")
    print(f"   First Name: {assessment.get_first_name()}")
    print(f"   Last Name:  {assessment.get_last_name()}")
    print(f"   Email:      {assessment.get_email()}")
    
    print("\n3. NON-ENCRYPTED FIELDS (stored as plain text):")
    print(f"   Field of Expertise: {expertise}")
    print(f"   Total Score: {score}")
    print(f"   Eligibility: {status}")
    
    print("\n" + "=" * 80 + "\n")

conn.close()

print("\nCONCLUSION:")
print("-" * 50)
print("✅ Personal data (name, email) IS ENCRYPTED in database")
print("✅ Django admin DECRYPTS data for authorized viewing")
print("✅ Non-sensitive data (expertise, scores) stored as plain text")
print("✅ This is the CORRECT and INTENDED behavior!")
print("\nThe admin interface is SUPPOSED to show decrypted data")
print("so administrators can read and manage submissions.")
print("=" * 80)