// Modern VisaPro Website JavaScript
class VisaProWebsite {
    constructor() {
        this.init();
        this.setupEventListeners();
        this.initAnimations();
        this.initChatbot();
    }

    init() {
        // Initialize components
        this.navbar = document.getElementById('navbar');
        this.navToggle = document.getElementById('nav-toggle');
        this.navMenu = document.getElementById('nav-menu');
        this.contactForm = document.getElementById('contactForm');
        
        // Set initial states
        this.isMenuOpen = false;
        this.scrollPosition = 0;
    }

    setupEventListeners() {
        // Navigation
        this.navToggle?.addEventListener('click', (e) => {
            e.preventDefault();
            e.stopPropagation();
            this.toggleMobileMenu();
        });
        
        // Smooth scrolling for navigation links
        document.querySelectorAll('a[href^="#"]').forEach(anchor => {
            anchor.addEventListener('click', (e) => this.handleSmoothScroll(e));
        });

        // Scroll events
        window.addEventListener('scroll', () => this.handleScroll());
        
        // Contact form
        this.contactForm?.addEventListener('submit', (e) => this.handleContactForm(e));
        
        // Close mobile menu when clicking outside
        document.addEventListener('click', (e) => this.handleOutsideClick(e));
        
        // Window resize
        window.addEventListener('resize', () => this.handleResize());
        
        // Prevent menu links from closing menu immediately (let smooth scroll handle it)
        document.querySelectorAll('.nav-link').forEach(link => {
            link.addEventListener('click', () => {
                // Close menu after a short delay to allow smooth scroll to start
                setTimeout(() => {
                    if (this.isMenuOpen) {
                        this.toggleMobileMenu();
                    }
                }, 100);
            });
        });
    }

    toggleMobileMenu() {
        this.isMenuOpen = !this.isMenuOpen;
        this.navMenu?.classList.toggle('active');
        this.navToggle?.classList.toggle('active');
        
        // Prevent body scroll when menu is open
        document.body.style.overflow = this.isMenuOpen ? 'hidden' : '';
    }

    handleSmoothScroll(e) {
        e.preventDefault();
        const targetId = e.currentTarget.getAttribute('href');
        const targetElement = document.querySelector(targetId);
        
        if (targetElement) {
            const offsetTop = targetElement.offsetTop - 80; // Account for fixed navbar
            
            window.scrollTo({
                top: offsetTop,
                behavior: 'smooth'
            });
            
            // Close mobile menu if open
            if (this.isMenuOpen) {
                this.toggleMobileMenu();
            }
            
            // Update active nav link
            this.updateActiveNavLink(targetId);
        }
    }

    updateActiveNavLink(targetId) {
        document.querySelectorAll('.nav-link').forEach(link => {
            link.classList.remove('active');
            if (link.getAttribute('href') === targetId) {
                link.classList.add('active');
            }
        });
    }

    handleScroll() {
        const currentScroll = window.pageYOffset;
        
        // Navbar scroll effect
        if (currentScroll > 100) {
            this.navbar?.classList.add('scrolled');
        } else {
            this.navbar?.classList.remove('scrolled');
        }
        
        // Update active section in navigation
        this.updateActiveSection();
        
        this.scrollPosition = currentScroll;
    }

    updateActiveSection() {
        const sections = document.querySelectorAll('section[id]');
        const scrollPos = window.pageYOffset + 150;
        
        sections.forEach(section => {
            const sectionTop = section.offsetTop;
            const sectionHeight = section.offsetHeight;
            const sectionId = section.getAttribute('id');
            
            if (scrollPos >= sectionTop && scrollPos < sectionTop + sectionHeight) {
                this.updateActiveNavLink(`#${sectionId}`);
            }
        });
    }

    handleContactForm(e) {
        e.preventDefault();
        
        const formData = new FormData(this.contactForm);
        const data = Object.fromEntries(formData);
        
        // Basic validation
        if (!this.validateForm(data)) {
            return;
        }
        
        // Show loading state
        const submitBtn = this.contactForm.querySelector('.submit-btn');
        const originalText = submitBtn.innerHTML;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Sending...';
        submitBtn.disabled = true;
        
        // Simulate form submission
        setTimeout(() => {
            this.showNotification('Thank you! Your message has been sent successfully. We\'ll get back to you soon.', 'success');
            this.contactForm.reset();
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        }, 2000);
    }

    validateForm(data) {
        const required = ['firstName', 'lastName', 'email', 'service', 'message'];
        const missing = required.filter(field => !data[field] || data[field].trim() === '');
        
        if (missing.length > 0) {
            this.showNotification('Please fill in all required fields.', 'error');
            return false;
        }
        
        // Email validation
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(data.email)) {
            this.showNotification('Please enter a valid email address.', 'error');
            return false;
        }
        
        return true;
    }

    showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <div class="notification-content">
                <i class="fas fa-${type === 'success' ? 'check-circle' : type === 'error' ? 'exclamation-circle' : 'info-circle'}"></i>
                <span>${message}</span>
            </div>
            <button class="notification-close">
                <i class="fas fa-times"></i>
            </button>
        `;
        
        // Add styles
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: ${type === 'success' ? '#10b981' : type === 'error' ? '#ef4444' : '#3b82f6'};
            color: white;
            padding: 1rem 1.5rem;
            border-radius: 12px;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.2);
            z-index: 10000;
            max-width: 400px;
            animation: slideInRight 0.3s ease-out;
        `;
        
        // Add to DOM
        document.body.appendChild(notification);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            notification.style.animation = 'slideOutRight 0.3s ease-out';
            setTimeout(() => notification.remove(), 300);
        }, 5000);
        
        // Manual close
        notification.querySelector('.notification-close').addEventListener('click', () => {
            notification.style.animation = 'slideOutRight 0.3s ease-out';
            setTimeout(() => notification.remove(), 300);
        });
    }

    handleOutsideClick(e) {
        if (this.isMenuOpen && !this.navMenu?.contains(e.target) && !this.navToggle?.contains(e.target)) {
            this.toggleMobileMenu();
        }
    }

    handleResize() {
        if (window.innerWidth > 768 && this.isMenuOpen) {
            this.toggleMobileMenu();
        }
    }

    initAnimations() {
        // Intersection Observer for animations
        const observerOptions = {
            threshold: 0.1,
            rootMargin: '0px 0px -50px 0px'
        };

        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    entry.target.classList.add('loaded');
                    
                    // Animate counters
                    if (entry.target.classList.contains('stat-number')) {
                        this.animateCounter(entry.target);
                    }
                }
            });
        }, observerOptions);

        // Observe elements for animation
        document.querySelectorAll('.loading, .stat-number').forEach(el => {
            el.classList.add('loading');
            observer.observe(el);
        });

        // Add loading class to animated elements
        document.querySelectorAll('.service-card, .feature-item, .contact-card, .visa-card, .visa-type-card, .testimonial-card').forEach(el => {
            el.classList.add('loading');
            observer.observe(el);
        });
        
        // Initialize visa tabs
        this.initVisaTabs();
        
        // Initialize testimonials slider
        this.initTestimonials();
    }
    
    initVisaTabs() {
        const tabButtons = document.querySelectorAll('.tab-btn');
        const tabPanes = document.querySelectorAll('.tab-pane');
        
        console.log('Initializing visa tabs:', tabButtons.length, 'buttons found');
        
        tabButtons.forEach(button => {
            button.addEventListener('click', (e) => {
                e.preventDefault();
                
                // Remove active class from all buttons and panes
                tabButtons.forEach(btn => btn.classList.remove('active'));
                tabPanes.forEach(pane => pane.classList.remove('active'));
                
                // Add active class to clicked button
                button.classList.add('active');
                
                // Show corresponding tab pane
                const targetTab = button.getAttribute('data-tab');
                console.log('Switching to tab:', targetTab);
                
                const targetPane = document.getElementById(targetTab);
                if (targetPane) {
                    targetPane.classList.add('active');
                } else {
                    console.error('Tab pane not found:', targetTab);
                }
            });
        });
    }
    
    initTestimonials() {
        this.currentTestimonial = 0;
        this.testimonials = document.querySelectorAll('.testimonial-slide');
        this.totalTestimonials = this.testimonials.length;
        
        if (this.totalTestimonials === 0) return;
        
        // Navigation buttons
        const prevBtn = document.getElementById('prevTestimonial');
        const nextBtn = document.getElementById('nextTestimonial');
        const dots = document.querySelectorAll('.dot');
        
        if (prevBtn) {
            prevBtn.addEventListener('click', () => this.prevTestimonial());
        }
        
        if (nextBtn) {
            nextBtn.addEventListener('click', () => this.nextTestimonial());
        }
        
        // Dot navigation
        dots.forEach((dot, index) => {
            dot.addEventListener('click', () => this.goToTestimonial(index));
        });
        
        // Auto-play testimonials
        this.testimonialInterval = setInterval(() => {
            this.nextTestimonial();
        }, 6000);
        
        // Pause auto-play on hover
        const testimonialContainer = document.querySelector('.testimonials-container');
        if (testimonialContainer) {
            testimonialContainer.addEventListener('mouseenter', () => {
                clearInterval(this.testimonialInterval);
            });
            
            testimonialContainer.addEventListener('mouseleave', () => {
                this.testimonialInterval = setInterval(() => {
                    this.nextTestimonial();
                }, 6000);
            });
        }
    }
    
    showTestimonial(index) {
        if (!this.testimonials || this.testimonials.length === 0) return;
        
        // Hide all testimonials
        this.testimonials.forEach(testimonial => {
            testimonial.classList.remove('active');
        });
        
        // Show target testimonial
        if (this.testimonials[index]) {
            this.testimonials[index].classList.add('active');
        }
        
        // Update dots
        const dots = document.querySelectorAll('.dot');
        dots.forEach((dot, i) => {
            dot.classList.toggle('active', i === index);
        });
        
        this.currentTestimonial = index;
    }
    
    nextTestimonial() {
        const nextIndex = (this.currentTestimonial + 1) % this.totalTestimonials;
        this.showTestimonial(nextIndex);
    }
    
    prevTestimonial() {
        const prevIndex = (this.currentTestimonial - 1 + this.totalTestimonials) % this.totalTestimonials;
        this.showTestimonial(prevIndex);
    }
    
    goToTestimonial(index) {
        this.showTestimonial(index);
    }

    animateCounter(element) {
        const target = parseInt(element.getAttribute('data-target'));
        const duration = 2000;
        const start = 0;
        const increment = target / (duration / 16);
        let current = start;

        const updateCounter = () => {
            current += increment;
            if (current < target) {
                element.textContent = Math.floor(current);
                requestAnimationFrame(updateCounter);
            } else {
                element.textContent = target;
            }
        };

        updateCounter();
    }

    initChatbot() {
        this.chatbot = new ModernChatbot();
    }
}

// Modern Chatbot Class
class ModernChatbot {
    constructor() {
        this.isOpen = false;
        this.messages = [];
        this.responses = {
            'visa-types': {
                text: "We offer comprehensive visa services including:\n\n• **Work Visas** (H1B, L1, O1)\n• **Family Immigration** (K1, Spouse visas)\n• **Student Visas** (F1, M1, J1)\n• **Green Card Services**\n• **Tourist & Business Visas**\n\nWhich type interests you most?",
                actions: ['Work Visa', 'Family Immigration', 'Student Visa', 'Green Card']
            },
            'consultation': {
                text: "Great choice! Our free consultation includes:\n\n• **Case Assessment** - Review your eligibility\n• **Strategy Planning** - Personalized roadmap\n• **Timeline Discussion** - Expected processing times\n• **Cost Breakdown** - Transparent pricing\n\nWould you like to schedule your consultation now?",
                actions: ['Schedule Now', 'Learn More', 'Call Us']
            },
            'contact': {
                text: "Here's how to reach our expert team:\n\n📞 **Phone:** +1 (555) 123-4567\n📧 **Email:** info@visapro.com\n📍 **Address:** 123 Immigration Ave, Suite 500, NY\n🕒 **Hours:** Mon-Fri 9AM-6PM, Sat 10AM-4PM\n\nHow would you prefer to connect?",
                actions: ['Call Now', 'Send Email', 'Visit Office']
            }
        };
        
        // Wait for DOM to be ready
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', () => this.init());
        } else {
            this.init();
        }
    }

    init() {
        // Wait a bit more to ensure all elements are rendered
        setTimeout(() => {
            this.trigger = document.getElementById('chatbotTrigger');
            this.window = document.getElementById('chatbotWindow');
            this.closeBtn = document.getElementById('closeChat');
            this.input = document.getElementById('chatInput');
            this.sendBtn = document.getElementById('sendMessage');
            this.messagesContainer = document.getElementById('chatbotMessages');
            
            console.log('Chatbot elements:', {
                trigger: !!this.trigger,
                window: !!this.window,
                closeBtn: !!this.closeBtn,
                input: !!this.input,
                sendBtn: !!this.sendBtn,
                messagesContainer: !!this.messagesContainer
            });
            
            if (this.trigger && this.window) {
                this.setupEventListeners();
                console.log('Chatbot initialized successfully');
            } else {
                console.error('Chatbot elements not found');
            }
        }, 100);
    }

    setupEventListeners() {
        // Trigger click
        if (this.trigger) {
            this.trigger.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                console.log('Chatbot trigger clicked');
                this.openChat();
            });
        }
        
        // Close button
        if (this.closeBtn) {
            this.closeBtn.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                console.log('Chatbot close clicked');
                this.closeChat();
            });
        }
        
        // Send button
        if (this.sendBtn) {
            this.sendBtn.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                this.sendMessage();
            });
        }
        
        // Input enter key
        if (this.input) {
            this.input.addEventListener('keypress', (e) => {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    this.sendMessage();
                }
            });
        }

        // Quick actions - use event delegation
        document.addEventListener('click', (e) => {
            if (e.target.closest('.quick-action')) {
                e.preventDefault();
                e.stopPropagation();
                const actionBtn = e.target.closest('.quick-action');
                const action = actionBtn.getAttribute('data-action');
                console.log('Quick action clicked:', action);
                this.handleQuickAction(action);
            }
        });
        
        // Prevent chatbot window from closing when clicking inside
        if (this.window) {
            this.window.addEventListener('click', (e) => {
                e.stopPropagation();
            });
        }
        
        // Close chatbot when clicking outside
        document.addEventListener('click', (e) => {
            if (this.isOpen && !this.window.contains(e.target) && !this.trigger.contains(e.target)) {
                this.closeChat();
            }
        });
    }

    openChat() {
        console.log('Opening chat...');
        this.isOpen = true;
        if (this.window) {
            this.window.classList.add('active');
            this.window.style.display = 'flex';
        }
        if (this.trigger) {
            this.trigger.style.display = 'none';
        }
        if (this.input) {
            setTimeout(() => this.input.focus(), 300);
        }
    }

    closeChat() {
        console.log('Closing chat...');
        this.isOpen = false;
        if (this.window) {
            this.window.classList.remove('active');
            setTimeout(() => {
                this.window.style.display = 'none';
            }, 300);
        }
        if (this.trigger) {
            this.trigger.style.display = 'flex';
        }
    }

    sendMessage() {
        if (!this.input) return;
        
        const message = this.input.value.trim();
        if (!message) return;

        console.log('Sending message:', message);
        this.addUserMessage(message);
        this.input.value = '';
        
        setTimeout(() => {
            this.showTypingIndicator();
            setTimeout(() => {
                this.hideTypingIndicator();
                this.generateBotResponse(message);
            }, 1500);
        }, 500);
    }

    addUserMessage(message) {
        if (!this.messagesContainer) return;
        
        const messageDiv = document.createElement('div');
        messageDiv.className = 'message user-message';
        messageDiv.innerHTML = `
            <div class="message-content">
                <p>${this.escapeHtml(message)}</p>
            </div>
        `;
        this.messagesContainer.appendChild(messageDiv);
        this.scrollToBottom();
    }

    addBotMessage(message, actions = []) {
        if (!this.messagesContainer) return;
        
        const messageDiv = document.createElement('div');
        messageDiv.className = 'message bot-message';
        
        let actionsHTML = '';
        if (actions.length > 0) {
            actionsHTML = `
                <div class="quick-actions">
                    ${actions.map(action => `
                        <button class="quick-action" data-action="${this.slugify(action)}">
                            <span>${this.escapeHtml(action)}</span>
                        </button>
                    `).join('')}
                </div>
            `;
        }

        messageDiv.innerHTML = `
            <div class="message-avatar">
                <img src="https://images.unsplash.com/photo-1494790108755-2616b612b786?ixlib=rb-4.0.3&auto=format&fit=crop&w=100&q=80" alt="Sarah">
            </div>
            <div class="message-content">
                <p>${this.formatMessage(message)}</p>
                ${actionsHTML}
            </div>
        `;
        this.messagesContainer.appendChild(messageDiv);
        this.scrollToBottom();
    }

    showTypingIndicator() {
        if (!this.messagesContainer) return;
        
        const typingDiv = document.createElement('div');
        typingDiv.className = 'message bot-message typing-message';
        typingDiv.innerHTML = `
            <div class="message-avatar">
                <img src="https://images.unsplash.com/photo-1494790108755-2616b612b786?ixlib=rb-4.0.3&auto=format&fit=crop&w=100&q=80" alt="Sarah">
            </div>
            <div class="message-content">
                <div style="background: white; padding: 12px 16px; border-radius: 18px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <span style="font-size: 0.9rem; color: #666;">Sarah is typing</span>
                        <div style="display: flex; gap: 3px;">
                            <div style="width: 6px; height: 6px; background: #ccc; border-radius: 50%; animation: typing 1.4s infinite;"></div>
                            <div style="width: 6px; height: 6px; background: #ccc; border-radius: 50%; animation: typing 1.4s infinite 0.2s;"></div>
                            <div style="width: 6px; height: 6px; background: #ccc; border-radius: 50%; animation: typing 1.4s infinite 0.4s;"></div>
                        </div>
                    </div>
                </div>
            </div>
        `;
        this.messagesContainer.appendChild(typingDiv);
        this.scrollToBottom();
    }

    hideTypingIndicator() {
        if (!this.messagesContainer) return;
        
        const typingMessage = this.messagesContainer.querySelector('.typing-message');
        if (typingMessage) {
            typingMessage.remove();
        }
    }

    generateBotResponse(userMessage) {
        const message = userMessage.toLowerCase();
        let response = {
            text: "Thank you for your message! I'd be happy to help you with your immigration needs. Our experienced team can assist with all types of visas and immigration matters. Would you like to speak with one of our specialists?",
            actions: ['Schedule Consultation', 'Learn More', 'Contact Us']
        };

        // Simple keyword matching
        if (message.includes('visa') || message.includes('type')) {
            response = this.responses['visa-types'];
        } else if (message.includes('consultation') || message.includes('consult') || message.includes('appointment')) {
            response = this.responses['consultation'];
        } else if (message.includes('contact') || message.includes('phone') || message.includes('email') || message.includes('address')) {
            response = this.responses['contact'];
        } else if (message.includes('h1b') || message.includes('work')) {
            response = {
                text: "H1B work visas are perfect for skilled professionals! Here's what you need to know:\n\n• **Duration:** 3 years initially, renewable up to 6 years\n• **Requirements:** Bachelor's degree + job offer\n• **Process:** Employer petition + lottery system\n• **Benefits:** Dual intent allowed\n\nWould you like help with your H1B application?",
                actions: ['Start Application', 'Check Eligibility', 'Learn More']
            };
        } else if (message.includes('family') || message.includes('spouse') || message.includes('marriage')) {
            response = {
                text: "Family immigration helps reunite loved ones! We handle:\n\n• **K1 Fiancé Visas** - For engaged couples\n• **Spouse Visas** - For married couples\n• **Parent/Child** - Family reunification\n• **Green Cards** - Permanent residency\n\nWhat's your family situation?",
                actions: ['Spouse Visa', 'Fiancé Visa', 'Family Petition']
            };
        } else if (message.includes('student') || message.includes('f1') || message.includes('study')) {
            response = {
                text: "Student visas open doors to world-class education! Options include:\n\n• **F1 Visa** - Academic studies at universities\n• **M1 Visa** - Vocational/technical training\n• **J1 Visa** - Exchange programs\n• **OPT/CPT** - Work authorization for students\n\nWhat type of studies are you planning?",
                actions: ['F1 Visa', 'University Programs', 'Work Authorization']
            };
        } else if (message.includes('green card') || message.includes('permanent')) {
            response = {
                text: "Green Cards provide permanent residency! Pathways include:\n\n• **Employment-Based** - Through job sponsorship\n• **Family-Based** - Through relatives\n• **Investment** - EB5 investor program\n• **Diversity Lottery** - Annual program\n\nWhich category might apply to you?",
                actions: ['Employment Based', 'Family Based', 'Investment']
            };
        } else if (message.includes('hello') || message.includes('hi') || message.includes('hey')) {
            response = {
                text: "Hello! Welcome to VisaPro! 👋\n\nI'm Sarah, your immigration specialist. I'm here to help you navigate the complex world of U.S. immigration with confidence.\n\nHow can I assist you today?",
                actions: ['Visa Types', 'Free Consultation', 'Contact Info']
            };
        } else if (message.includes('thank') || message.includes('thanks')) {
            response = {
                text: "You're very welcome! 😊\n\nI'm here whenever you need assistance with your immigration journey. Is there anything else I can help you with today?",
                actions: ['Ask Another Question', 'Schedule Consultation', 'Contact Attorney']
            };
        }

        this.addBotMessage(response.text, response.actions);
    }

    handleQuickAction(action) {
        if (!action) return;
        
        const actionText = action.replace(/-/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
        this.addUserMessage(actionText);
        
        setTimeout(() => {
            this.showTypingIndicator();
            setTimeout(() => {
                this.hideTypingIndicator();
                this.handleSpecificAction(action);
            }, 1000);
        }, 300);
    }

    handleSpecificAction(action) {
        let response = {
            text: "Thank you for your interest! Our expert team is ready to help you with that. Would you like to schedule a consultation to discuss your specific needs?",
            actions: ['Schedule Now', 'Learn More', 'Contact Us']
        };

        switch (action) {
            case 'schedule-now':
            case 'schedule-consultation':
                response = {
                    text: "Perfect! Let's get your consultation scheduled. You can:\n\n📞 **Call us directly:** +1 (555) 123-4567\n📅 **Online booking:** Available 24/7\n📧 **Email us:** consultation@visapro.com\n\nOur consultations are completely free and confidential. What works best for you?",
                    actions: ['Call Now', 'Email Us', 'Online Booking']
                };
                break;
            case 'call-now':
                response = {
                    text: "Great choice! You can reach us at:\n\n📞 **+1 (555) 123-4567**\n\n**Office Hours:**\n• Monday - Friday: 9:00 AM - 6:00 PM\n• Saturday: 10:00 AM - 4:00 PM\n• Sunday: Closed\n\nOur immigration specialists are standing by to help!",
                    actions: ['Send Email Instead', 'Office Hours', 'Emergency Contact']
                };
                break;
            case 'send-email':
            case 'email-us':
                response = {
                    text: "Excellent! You can email us at:\n\n📧 **info@visapro.com**\n📧 **consultation@visapro.com** (for consultations)\n\n**Response Time:** We typically respond within 4-6 hours during business days.\n\n**For faster service:** Include your phone number and preferred contact time.",
                    actions: ['Call Instead', 'Urgent Matter', 'More Contact Options']
                };
                break;
            default:
                if (this.responses[action]) {
                    const resp = this.responses[action];
                    response = { text: resp.text, actions: resp.actions };
                }
        }

        this.addBotMessage(response.text, response.actions);
    }

    scrollToBottom() {
        if (this.messagesContainer) {
            this.messagesContainer.scrollTop = this.messagesContainer.scrollHeight;
        }
    }
    
    // Utility methods
    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    formatMessage(message) {
        return message
            .replace(/\n/g, '<br>')
            .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>');
    }
    
    slugify(text) {
        return text.toLowerCase().replace(/\s+/g, '-').replace(/[^\w-]/g, '');
    }
}

// Initialize the website when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    new VisaProWebsite();
    
    // Add CSS animations
    const style = document.createElement('style');
    style.textContent = `
        @keyframes slideInRight {
            from { transform: translateX(100%); opacity: 0; }
            to { transform: translateX(0); opacity: 1; }
        }
        
        @keyframes slideOutRight {
            from { transform: translateX(0); opacity: 1; }
            to { transform: translateX(100%); opacity: 0; }
        }
        
        @keyframes typing {
            0%, 60%, 100% { transform: translateY(0); }
            30% { transform: translateY(-10px); }
        }
        
        .notification-content {
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }
        
        .notification-close {
            background: none;
            border: none;
            color: white;
            cursor: pointer;
            padding: 0.25rem;
            margin-left: 1rem;
            border-radius: 4px;
            transition: background 0.2s;
        }
        
        .notification-close:hover {
            background: rgba(255, 255, 255, 0.2);
        }
    `;
    document.head.appendChild(style);
});

// Add some utility functions
window.VisaProUtils = {
    // Smooth scroll to element
    scrollTo: (elementId) => {
        const element = document.getElementById(elementId);
        if (element) {
            element.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }
    },
    
    // Format phone number
    formatPhone: (phone) => {
        const cleaned = phone.replace(/\D/g, '');
        const match = cleaned.match(/^(\d{3})(\d{3})(\d{4})$/);
        return match ? `(${match[1]}) ${match[2]}-${match[3]}` : phone;
    },
    
    // Validate email
    isValidEmail: (email) => {
        return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
    }
};

// FAQ Accordion functionality
function initFAQ() {
    const faqItems = document.querySelectorAll('.faq-item');
    
    faqItems.forEach(item => {
        const question = item.querySelector('.faq-question');
        
        question.addEventListener('click', () => {
            const isActive = item.classList.contains('active');
            
            // Close all other FAQ items
            faqItems.forEach(otherItem => {
                if (otherItem !== item) {
                    otherItem.classList.remove('active');
                }
            });
            
            // Toggle current item
            if (isActive) {
                item.classList.remove('active');
            } else {
                item.classList.add('active');
            }
        });
    });
}

// Newsletter form handling
function initNewsletter() {
    const newsletterForm = document.getElementById('newsletterForm');
    
    if (newsletterForm) {
        newsletterForm.addEventListener('submit', (e) => {
            e.preventDefault();
            
            const email = newsletterForm.querySelector('input[type="email"]').value;
            
            // Show success message
            showNotification('Success! You\'ve been subscribed to our newsletter.', 'success');
            
            // Reset form
            newsletterForm.reset();
            
            // In a real application, you would send this to your backend
            console.log('Newsletter subscription:', email);
        });
    }
}

// Notification system
function showNotification(message, type = 'info') {
    // Remove existing notifications
    const existingNotification = document.querySelector('.notification');
    if (existingNotification) {
        existingNotification.remove();
    }

    // Create notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <i class="fas fa-${type === 'success' ? 'check-circle' : 'info-circle'}"></i>
            <span>${message}</span>
        </div>
        <button class="notification-close">
            <i class="fas fa-times"></i>
        </button>
    `;

    // Add to body
    document.body.appendChild(notification);

    // Animate in
    setTimeout(() => notification.classList.add('show'), 10);

    // Close button
    notification.querySelector('.notification-close').addEventListener('click', () => {
        notification.classList.remove('show');
        setTimeout(() => notification.remove(), 300);
    });

    // Auto remove after 5 seconds
    setTimeout(() => {
        notification.classList.remove('show');
        setTimeout(() => notification.remove(), 300);
    }, 5000);
}

// Scroll animations for timeline and other elements
function initScrollAnimations() {
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -100px 0px'
    };

        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    entry.target.classList.add('animate-in');
                }
            });
        }, observerOptions);

        // Observe timeline items
        document.querySelectorAll('.timeline-item').forEach(item => {
            observer.observe(item);
        });

        // Observe country cards
        document.querySelectorAll('.country-card').forEach(card => {
            observer.observe(card);
        });

        // Observe FAQ items
        document.querySelectorAll('.faq-item').forEach(item => {
            observer.observe(item);
        });
}

// Counter animation for stats
function animateCounters() {
    const counters = document.querySelectorAll('.stat-number');
    
    counters.forEach(counter => {
        const target = parseInt(counter.getAttribute('data-target'));
        const duration = 2000; // 2 seconds
        const increment = target / (duration / 16); // 60fps
        let current = 0;

        const updateCounter = () => {
            current += increment;
            if (current < target) {
                counter.textContent = Math.floor(current);
                requestAnimationFrame(updateCounter);
            } else {
                counter.textContent = target;
            }
        };

        // Start animation when element is in viewport
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting && counter.textContent === '0') {
                    updateCounter();
                }
            });
        }, { threshold: 0.5 });

        observer.observe(counter);
    });
}

// Enhanced initialization
function initEnhancedFeatures() {
    initFAQ();
    initNewsletter();
    initScrollAnimations();
    animateCounters();
}

// Add notification styles dynamically
const notificationStyles = document.createElement('style');
notificationStyles.textContent = `
    .notification {
        position: fixed;
        top: 100px;
        right: 20px;
        background: white;
        padding: 1rem 1.5rem;
        border-radius: 12px;
        box-shadow: 0 10px 40px rgba(0, 0, 0, 0.15);
        display: flex;
        align-items: center;
        gap: 1rem;
        z-index: 10000;
        transform: translateX(400px);
        transition: transform 0.3s ease-out;
        max-width: 400px;
    }

    .notification.show {
        transform: translateX(0);
    }

    .notification-content {
        display: flex;
        align-items: center;
        gap: 0.75rem;
        flex: 1;
    }

    .notification-content i {
        font-size: 1.5rem;
    }

    .notification-success {
        border-left: 4px solid #11998e;
    }

    .notification-success .notification-content i {
        color: #11998e;
    }

    .notification-info {
        border-left: 4px solid #4facfe;
    }

    .notification-info .notification-content i {
        color: #4facfe;
    }

    .notification-close {
        background: none;
        border: none;
        cursor: pointer;
        color: #718096;
        font-size: 1.25rem;
        padding: 0.25rem;
        transition: color 0.2s;
    }

    .notification-close:hover {
        color: #1a202c;
    }

    @media (max-width: 768px) {
        .notification {
            right: 10px;
            left: 10px;
            max-width: none;
        }
    }
`;
document.head.appendChild(notificationStyles);

// Initialize website when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    const website = new VisaProWebsite();
    initEnhancedFeatures();
});


// Modal and Floating Action Buttons functionality
class EnhancedFeatures {
    constructor() {
        this.initModal();
        this.initFloatingButtons();
        this.initScrollToTop();
    }

    initModal() {
        const modal = document.getElementById('consultationModal');
        const quickConsultBtn = document.getElementById('quickConsultation');
        const closeModalBtn = document.getElementById('closeModal');
        const modalOverlay = modal?.querySelector('.modal-overlay');
        const consultForm = document.getElementById('quickConsultationForm');

        // Open modal
        quickConsultBtn?.addEventListener('click', () => {
            modal?.classList.add('active');
            document.body.style.overflow = 'hidden';
        });

        // Close modal
        const closeModal = () => {
            modal?.classList.remove('active');
            document.body.style.overflow = '';
        };

        closeModalBtn?.addEventListener('click', closeModal);
        modalOverlay?.addEventListener('click', closeModal);

        // Handle form submission
        consultForm?.addEventListener('submit', (e) => {
            e.preventDefault();
            
            const formData = {
                name: document.getElementById('consultName').value,
                email: document.getElementById('consultEmail').value,
                phone: document.getElementById('consultPhone').value,
                visaType: document.getElementById('consultVisa').value,
                date: document.getElementById('consultDate').value,
                time: document.getElementById('consultTime').value
            };

            console.log('Consultation booking:', formData);
            
            // Show success notification
            showNotification('Consultation booked successfully! We\'ll contact you soon.', 'success');
            
            // Close modal and reset form
            closeModal();
            consultForm.reset();
        });

        // Close modal on Escape key
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape' && modal?.classList.contains('active')) {
                closeModal();
            }
        });
    }

    initFloatingButtons() {
        const whatsappBtn = document.getElementById('whatsappButton');
        const callBtn = document.getElementById('callButton');

        whatsappBtn?.addEventListener('click', () => {
            window.open('https://wa.me/15551234567?text=Hi, I need help with my visa application', '_blank');
        });

        callBtn?.addEventListener('click', () => {
            window.location.href = 'tel:+15551234567';
        });
    }

    initScrollToTop() {
        // Create scroll to top button
        const scrollBtn = document.createElement('button');
        scrollBtn.className = 'scroll-to-top';
        scrollBtn.innerHTML = '<i class="fas fa-arrow-up"></i>';
        scrollBtn.setAttribute('aria-label', 'Scroll to top');
        document.body.appendChild(scrollBtn);

        // Show/hide button based on scroll position
        window.addEventListener('scroll', () => {
            if (window.pageYOffset > 300) {
                scrollBtn.classList.add('visible');
            } else {
                scrollBtn.classList.remove('visible');
            }
        });

        // Scroll to top on click
        scrollBtn.addEventListener('click', () => {
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        });
    }

    showNotification(message, type = 'info') {
        const existingNotification = document.querySelector('.notification');
        if (existingNotification) {
            existingNotification.remove();
        }

        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <div class="notification-content">
                <i class="fas fa-${type === 'success' ? 'check-circle' : 'info-circle'}"></i>
                <span>${message}</span>
            </div>
            <button class="notification-close">
                <i class="fas fa-times"></i>
            </button>
        `;

        document.body.appendChild(notification);
        setTimeout(() => notification.classList.add('show'), 10);

        notification.querySelector('.notification-close').addEventListener('click', () => {
            notification.classList.remove('show');
            setTimeout(() => notification.remove(), 300);
        });

        setTimeout(() => {
            notification.classList.remove('show');
            setTimeout(() => notification.remove(), 300);
        }, 5000);
    }
}

// Initialize enhanced features when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    new EnhancedFeatures();
});

// Add pulse animation to consultation button
document.addEventListener('DOMContentLoaded', () => {
    const consultBtn = document.querySelector('.btn-consultation');
    if (consultBtn) {
        consultBtn.classList.add('pulse-animation');
    }
});

// Lazy loading for images
document.addEventListener('DOMContentLoaded', () => {
    const images = document.querySelectorAll('img[data-src]');
    
    const imageObserver = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const img = entry.target;
                img.src = img.dataset.src;
                img.removeAttribute('data-src');
                observer.unobserve(img);
            }
        });
    });

    images.forEach(img => imageObserver.observe(img));
});

// Add smooth reveal animations
document.addEventListener('DOMContentLoaded', () => {
    const revealElements = document.querySelectorAll('.service-card, .visa-type-card, .testimonial-card');
    
    const revealObserver = new IntersectionObserver((entries) => {
        entries.forEach((entry, index) => {
            if (entry.isIntersecting) {
                setTimeout(() => {
                    entry.target.style.opacity = '1';
                    entry.target.style.transform = 'translateY(0)';
                }, index * 100);
            }
        });
    }, { threshold: 0.1 });

    revealElements.forEach(el => {
        el.style.opacity = '0';
        el.style.transform = 'translateY(30px)';
        el.style.transition = 'opacity 0.6s ease-out, transform 0.6s ease-out';
        revealObserver.observe(el);
    });
});


// Modern UI Enhancements
class ModernUIEnhancements {
    constructor() {
        this.initParallax();
        this.initCursorEffect();
        this.initSmoothReveal();
        this.initCardTilt();
    }

    // Parallax scrolling effect
    initParallax() {
        window.addEventListener('scroll', () => {
            const scrolled = window.pageYOffset;
            const parallaxElements = document.querySelectorAll('.hero-decoration, .floating-shape');
            
            parallaxElements.forEach((el, index) => {
                const speed = 0.5 + (index * 0.1);
                el.style.transform = `translateY(${scrolled * speed}px)`;
            });
        });
    }

    // Custom cursor effect for interactive elements
    initCursorEffect() {
        const interactiveElements = document.querySelectorAll('button, a, .service-card, .country-card');
        
        interactiveElements.forEach(el => {
            el.addEventListener('mouseenter', () => {
                document.body.style.cursor = 'pointer';
            });
            
            el.addEventListener('mouseleave', () => {
                document.body.style.cursor = 'default';
            });
        });
    }

    // Smooth reveal animations on scroll
    initSmoothReveal() {
        const revealElements = document.querySelectorAll('.service-card, .feature-item, .country-card, .faq-item, .timeline-item');
        
        const revealObserver = new IntersectionObserver((entries) => {
            entries.forEach((entry, index) => {
                if (entry.isIntersecting) {
                    setTimeout(() => {
                        entry.target.style.opacity = '1';
                        entry.target.style.transform = 'translateY(0) scale(1)';
                    }, index * 50);
                    revealObserver.unobserve(entry.target);
                }
            });
        }, {
            threshold: 0.1,
            rootMargin: '0px 0px -50px 0px'
        });

        revealElements.forEach(el => {
            el.style.opacity = '0';
            el.style.transform = 'translateY(30px) scale(0.95)';
            el.style.transition = 'opacity 0.6s cubic-bezier(0.4, 0, 0.2, 1), transform 0.6s cubic-bezier(0.4, 0, 0.2, 1)';
            revealObserver.observe(el);
        });
    }

    // 3D tilt effect for cards
    initCardTilt() {
        const cards = document.querySelectorAll('.service-card, .country-card, .stat-item');
        
        cards.forEach(card => {
            card.addEventListener('mousemove', (e) => {
                const rect = card.getBoundingClientRect();
                const x = e.clientX - rect.left;
                const y = e.clientY - rect.top;
                
                const centerX = rect.width / 2;
                const centerY = rect.height / 2;
                
                const rotateX = (y - centerY) / 10;
                const rotateY = (centerX - x) / 10;
                
                card.style.transform = `perspective(1000px) rotateX(${rotateX}deg) rotateY(${rotateY}deg) scale(1.02)`;
            });
            
            card.addEventListener('mouseleave', () => {
                card.style.transform = 'perspective(1000px) rotateX(0) rotateY(0) scale(1)';
            });
        });
    }
}

// Gradient animation for hero background
class GradientAnimation {
    constructor() {
        this.initGradientShift();
    }

    initGradientShift() {
        const hero = document.querySelector('.hero');
        if (!hero) return;

        let hue = 0;
        setInterval(() => {
            hue = (hue + 0.5) % 360;
            hero.style.filter = `hue-rotate(${hue}deg)`;
        }, 100);
    }
}

// Typing effect for hero title
class TypingEffect {
    constructor(element, text, speed = 100) {
        this.element = element;
        this.text = text;
        this.speed = speed;
        this.index = 0;
    }

    type() {
        if (this.index < this.text.length) {
            this.element.textContent += this.text.charAt(this.index);
            this.index++;
            setTimeout(() => this.type(), this.speed);
        }
    }
}

// Magnetic button effect
class MagneticButtons {
    constructor() {
        this.initMagneticEffect();
    }

    initMagneticEffect() {
        const buttons = document.querySelectorAll('.btn-primary, .btn-secondary, .btn-consultation');
        
        buttons.forEach(button => {
            button.addEventListener('mousemove', (e) => {
                const rect = button.getBoundingClientRect();
                const x = e.clientX - rect.left - rect.width / 2;
                const y = e.clientY - rect.top - rect.height / 2;
                
                button.style.transform = `translate(${x * 0.2}px, ${y * 0.2}px)`;
            });
            
            button.addEventListener('mouseleave', () => {
                button.style.transform = 'translate(0, 0)';
            });
        });
    }
}

// Progress bar animation
class ProgressAnimation {
    constructor() {
        this.initProgressBars();
    }

    initProgressBars() {
        const progressBars = document.querySelectorAll('.progress-bar');
        
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const bar = entry.target;
                    const width = bar.style.width;
                    bar.style.width = '0';
                    setTimeout(() => {
                        bar.style.width = width;
                        bar.style.transition = 'width 1.5s cubic-bezier(0.4, 0, 0.2, 1)';
                    }, 100);
                    observer.unobserve(bar);
                }
            });
        }, { threshold: 0.5 });

        progressBars.forEach(bar => observer.observe(bar));
    }
}

// Particle effect for hero section
class ParticleEffect {
    constructor(container) {
        this.container = container;
        this.particles = [];
        this.init();
    }

    init() {
        if (!this.container) return;

        const canvas = document.createElement('canvas');
        canvas.style.position = 'absolute';
        canvas.style.top = '0';
        canvas.style.left = '0';
        canvas.style.width = '100%';
        canvas.style.height = '100%';
        canvas.style.pointerEvents = 'none';
        canvas.style.opacity = '0.3';
        
        this.container.appendChild(canvas);
        
        const ctx = canvas.getContext('2d');
        canvas.width = this.container.offsetWidth;
        canvas.height = this.container.offsetHeight;

        // Create particles
        for (let i = 0; i < 50; i++) {
            this.particles.push({
                x: Math.random() * canvas.width,
                y: Math.random() * canvas.height,
                radius: Math.random() * 2 + 1,
                vx: Math.random() * 0.5 - 0.25,
                vy: Math.random() * 0.5 - 0.25
            });
        }

        // Animate particles
        const animate = () => {
            ctx.clearRect(0, 0, canvas.width, canvas.height);
            
            this.particles.forEach(particle => {
                particle.x += particle.vx;
                particle.y += particle.vy;

                if (particle.x < 0 || particle.x > canvas.width) particle.vx *= -1;
                if (particle.y < 0 || particle.y > canvas.height) particle.vy *= -1;

                ctx.beginPath();
                ctx.arc(particle.x, particle.y, particle.radius, 0, Math.PI * 2);
                ctx.fillStyle = 'rgba(99, 102, 241, 0.5)';
                ctx.fill();
            });

            requestAnimationFrame(animate);
        };

        animate();
    }
}

// Initialize all modern UI enhancements
document.addEventListener('DOMContentLoaded', () => {
    new ModernUIEnhancements();
    new MagneticButtons();
    new ProgressAnimation();
    
    // Add particle effect to hero
    const hero = document.querySelector('.hero');
    if (hero) {
        new ParticleEffect(hero);
    }
    
    // Add smooth scroll behavior
    document.documentElement.style.scrollBehavior = 'smooth';
});

// Add loading animation
window.addEventListener('load', () => {
    document.body.classList.add('loaded');
    
    // Animate elements on load
    const animateOnLoad = document.querySelectorAll('.hero-badge, .hero-title, .hero-description, .hero-actions');
    animateOnLoad.forEach((el, index) => {
        setTimeout(() => {
            el.style.opacity = '1';
            el.style.transform = 'translateY(0)';
        }, index * 100);
    });
});


// Responsive Utilities
class ResponsiveHandler {
    constructor() {
        this.init();
        this.handleResize();
        window.addEventListener('resize', () => this.handleResize());
    }

    init() {
        this.isMobile = window.innerWidth < 768;
        this.isTablet = window.innerWidth >= 768 && window.innerWidth < 1024;
        this.isDesktop = window.innerWidth >= 1024;
    }

    handleResize() {
        this.init();
        
        // Close mobile menu on resize to desktop
        if (this.isDesktop) {
            const navMenu = document.getElementById('nav-menu');
            const navToggle = document.getElementById('nav-toggle');
            
            if (navMenu?.classList.contains('active')) {
                navMenu.classList.remove('active');
                navToggle?.classList.remove('active');
                document.body.style.overflow = '';
            }
        }

        // Adjust chatbot window size
        this.adjustChatbotSize();
        
        // Adjust modal size
        this.adjustModalSize();
    }

    adjustChatbotSize() {
        const chatbotWindow = document.getElementById('chatbotWindow');
        if (!chatbotWindow) return;

        if (this.isMobile) {
            chatbotWindow.style.width = '100%';
            chatbotWindow.style.height = '100%';
            chatbotWindow.style.borderRadius = '0';
        } else {
            chatbotWindow.style.width = '400px';
            chatbotWindow.style.height = '600px';
            chatbotWindow.style.borderRadius = 'var(--radius-2xl)';
        }
    }

    adjustModalSize() {
        const modalContent = document.querySelector('.modal-content');
        if (!modalContent) return;

        if (this.isMobile) {
            modalContent.style.margin = '1rem';
            modalContent.style.maxHeight = '85vh';
        } else {
            modalContent.style.margin = '2rem';
            modalContent.style.maxHeight = '90vh';
        }
    }
}

// Touch Gestures for Mobile
class TouchHandler {
    constructor() {
        this.initSwipeGestures();
    }

    initSwipeGestures() {
        let touchStartX = 0;
        let touchEndX = 0;

        // Swipe to close mobile menu
        const navMenu = document.getElementById('nav-menu');
        
        if (navMenu) {
            navMenu.addEventListener('touchstart', (e) => {
                touchStartX = e.changedTouches[0].screenX;
            }, { passive: true });

            navMenu.addEventListener('touchend', (e) => {
                touchEndX = e.changedTouches[0].screenX;
                this.handleSwipe(touchStartX, touchEndX);
            }, { passive: true });
        }
    }

    handleSwipe(startX, endX) {
        const swipeThreshold = 100;
        const diff = startX - endX;

        // Swipe left to close menu
        if (diff > swipeThreshold) {
            const navMenu = document.getElementById('nav-menu');
            const navToggle = document.getElementById('nav-toggle');
            
            if (navMenu?.classList.contains('active')) {
                navMenu.classList.remove('active');
                navToggle?.classList.remove('active');
                document.body.style.overflow = '';
            }
        }
    }
}

// Lazy Loading Images
class LazyLoader {
    constructor() {
        this.initLazyLoading();
    }

    initLazyLoading() {
        const images = document.querySelectorAll('img[data-src]');
        
        if ('IntersectionObserver' in window) {
            const imageObserver = new IntersectionObserver((entries, observer) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const img = entry.target;
                        img.src = img.dataset.src;
                        img.removeAttribute('data-src');
                        img.classList.add('loaded');
                        observer.unobserve(img);
                    }
                });
            }, {
                rootMargin: '50px'
            });

            images.forEach(img => imageObserver.observe(img));
        } else {
            // Fallback for older browsers
            images.forEach(img => {
                img.src = img.dataset.src;
                img.removeAttribute('data-src');
            });
        }
    }
}

// Performance Optimization
class PerformanceOptimizer {
    constructor() {
        this.optimizeAnimations();
        this.deferNonCriticalCSS();
    }

    optimizeAnimations() {
        // Disable animations on low-end devices
        const isLowEndDevice = navigator.hardwareConcurrency && navigator.hardwareConcurrency < 4;
        
        if (isLowEndDevice) {
            document.documentElement.style.setProperty('--transition-fast', '0s');
            document.documentElement.style.setProperty('--transition-base', '0s');
            document.documentElement.style.setProperty('--transition-slow', '0s');
        }
    }

    deferNonCriticalCSS() {
        // This would be used for loading non-critical CSS
        // Already handled in HTML with proper link tags
    }
}

// Viewport Height Fix for Mobile Browsers
class ViewportFix {
    constructor() {
        this.setViewportHeight();
        window.addEventListener('resize', () => this.setViewportHeight());
    }

    setViewportHeight() {
        // Fix for mobile browsers where 100vh includes address bar
        const vh = window.innerHeight * 0.01;
        document.documentElement.style.setProperty('--vh', `${vh}px`);
    }
}

// Orientation Change Handler
class OrientationHandler {
    constructor() {
        this.handleOrientation();
        window.addEventListener('orientationchange', () => {
            setTimeout(() => this.handleOrientation(), 200);
        });
    }

    handleOrientation() {
        const isLandscape = window.innerWidth > window.innerHeight;
        document.body.classList.toggle('landscape', isLandscape);
        document.body.classList.toggle('portrait', !isLandscape);
    }
}

// Initialize all responsive handlers
document.addEventListener('DOMContentLoaded', () => {
    new ResponsiveHandler();
    new TouchHandler();
    new LazyLoader();
    new PerformanceOptimizer();
    new ViewportFix();
    new OrientationHandler();
});

// Prevent zoom on double tap for iOS
let lastTouchEnd = 0;
document.addEventListener('touchend', (event) => {
    const now = Date.now();
    if (now - lastTouchEnd <= 300) {
        event.preventDefault();
    }
    lastTouchEnd = now;
}, { passive: false });

// Handle window resize with debounce
let resizeTimer;
window.addEventListener('resize', () => {
    clearTimeout(resizeTimer);
    resizeTimer = setTimeout(() => {
        // Trigger custom resize event
        window.dispatchEvent(new Event('optimizedResize'));
    }, 250);
});

// Smooth scroll polyfill for older browsers
if (!('scrollBehavior' in document.documentElement.style)) {
    const smoothScrollPolyfill = () => {
        const links = document.querySelectorAll('a[href^="#"]');
        
        links.forEach(link => {
            link.addEventListener('click', function(e) {
                e.preventDefault();
                const targetId = this.getAttribute('href');
                const targetElement = document.querySelector(targetId);
                
                if (targetElement) {
                    const offsetTop = targetElement.offsetTop - 80;
                    window.scrollTo({
                        top: offsetTop,
                        behavior: 'smooth'
                    });
                }
            });
        });
    };
    
    smoothScrollPolyfill();
}

// Add loading class to body
window.addEventListener('load', () => {
    document.body.classList.add('loaded');
});

// Service Worker Registration (for PWA capabilities)
if ('serviceWorker' in navigator) {
    window.addEventListener('load', () => {
        // Uncomment when you have a service worker file
        // navigator.serviceWorker.register('/sw.js')
        //     .then(registration => console.log('SW registered'))
        //     .catch(err => console.log('SW registration failed'));
    });
}
