"""
Test script to verify CRM views and lead capture integration
Run with: python manage.py shell < test_crm_views.py
"""

from django.contrib.auth.models import User
from crm.models import UserProfile, Lead, Service, Project
from main.models import Contact
from assessment.models import ProfileAssessment

print("\n" + "="*60)
print("CRM SYSTEM VIEWS TEST")
print("="*60)

# Test 1: Check if CRM app is properly configured
print("\n1. Testing CRM App Configuration...")
try:
    from crm import views
    print("   ✓ CRM views module loaded successfully")
except Exception as e:
    print(f"   ✗ Error loading CRM views: {e}")

# Test 2: Check URL configuration
print("\n2. Testing URL Configuration...")
try:
    from django.urls import reverse
    
    urls_to_test = [
        'client_login',
        'client_register',
        'client_dashboard',
        'staff_login',
        'admin_dashboard'
    ]
    
    for url_name in urls_to_test:
        try:
            url = reverse(url_name)
            print(f"   ✓ {url_name}: {url}")
        except Exception as e:
            print(f"   ✗ {url_name}: {e}")
            
except Exception as e:
    print(f"   ✗ Error testing URLs: {e}")

# Test 3: Check if leads are being created from forms
print("\n3. Testing Lead Creation from Forms...")

# Count existing leads
initial_lead_count = Lead.objects.count()
print(f"   Initial lead count: {initial_lead_count}")

# Check if any leads exist from contact forms
contact_leads = Lead.objects.filter(source='contact').count()
print(f"   Leads from contact forms: {contact_leads}")

# Check if any leads exist from assessments
assessment_leads = Lead.objects.filter(source='assessment').count()
print(f"   Leads from assessments: {assessment_leads}")

# Test 4: Check user roles
print("\n4. Testing User Roles...")
try:
    # Count users by role
    for role_code, role_name in UserProfile.ROLE_CHOICES:
        count = UserProfile.objects.filter(role=role_code).count()
        print(f"   {role_name}: {count} users")
except Exception as e:
    print(f"   ✗ Error checking roles: {e}")

# Test 5: Check services
print("\n5. Testing Services...")
service_count = Service.objects.count()
print(f"   Total services: {service_count}")
active_services = Service.objects.filter(is_active=True).count()
print(f"   Active services: {active_services}")

# Test 6: Check projects
print("\n6. Testing Projects...")
project_count = Project.objects.count()
print(f"   Total projects: {project_count}")

if project_count > 0:
    for status_code, status_name in Project.STATUS_CHOICES:
        count = Project.objects.filter(status=status_code).count()
        if count > 0:
            print(f"   {status_name}: {count} projects")

# Test 7: Test lead-to-user email matching
print("\n7. Testing Email Matching...")
try:
    # Get all users with emails
    users_with_email = User.objects.exclude(email='')
    
    for user in users_with_email[:5]:  # Test first 5
        matching_leads = Lead.objects.filter(email=user.email).count()
        if matching_leads > 0:
            print(f"   ✓ User {user.username} ({user.email}): {matching_leads} matching leads")
    
    if users_with_email.count() == 0:
        print("   No users with email addresses found")
        
except Exception as e:
    print(f"   ✗ Error testing email matching: {e}")

# Summary
print("\n" + "="*60)
print("TEST SUMMARY")
print("="*60)
print(f"Total Users: {User.objects.count()}")
print(f"Total Leads: {Lead.objects.count()}")
print(f"Total Projects: {Project.objects.count()}")
print(f"Total Services: {Service.objects.count()}")
print("\n✓ CRM System is ready!")
print("\nNext Steps:")
print("1. Run migrations: python manage.py migrate")
print("2. Create superuser: python manage.py createsuperuser")
print("3. Start server: python manage.py runserver")
print("4. Visit http://127.0.0.1:8000/client/login/")
print("5. Visit http://127.0.0.1:8000/client/register/")
print("="*60 + "\n")
