"""
Test script for Lead Conversion and Staff Registration System
"""

import os
import django

os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'myproject.settings')
django.setup()

from django.contrib.auth.models import User
from crm.models import UserProfile, Lead, Project, Service
from django.utils import timezone

def test_lead_conversion_without_user():
    """Test that lead conversion doesn't create user accounts"""
    print("\n=== Testing Lead Conversion Without User Creation ===")
    
    # Create a test lead
    lead = Lead.objects.create(
        source='contact',
        email='testlead@example.com',
        first_name='Test',
        last_name='Lead',
        phone='1234567890',
        status='new'
    )
    print(f"✓ Created test lead: {lead.email}")
    
    # Create a test service
    service = Service.objects.create(
        name='Test Service',
        description='Test service for lead conversion',
        category='visa',
        price=1000.00
    )
    print(f"✓ Created test service: {service.name}")
    
    # Simulate lead conversion (without creating user)
    project = Project.objects.create(
        title=f"Project for {lead.get_full_name()}",
        description="Test project from lead conversion",
        client=None,  # No user account yet
        client_email=lead.email,  # Store email for later linking
        lead=lead,
        service=service,
        status='pending_payment',
        budget=service.price
    )
    
    lead.status = 'converted'
    lead.converted_to_project = project
    lead.converted_at = timezone.now()
    lead.save()
    
    print(f"✓ Converted lead to project without creating user account")
    print(f"  - Project ID: {project.project_id}")
    print(f"  - Client: {project.client} (should be None)")
    print(f"  - Client Email: {project.client_email}")
    
    # Verify no user was created
    user_exists = User.objects.filter(email=lead.email).exists()
    print(f"✓ User account exists: {user_exists} (should be False)")
    
    return lead, project

def test_client_registration_with_project_linking():
    """Test that client registration links existing projects"""
    print("\n=== Testing Client Registration with Project Linking ===")
    
    # Get the test lead and project from previous test
    lead = Lead.objects.filter(email='testlead@example.com').first()
    if not lead:
        print("✗ Test lead not found. Run test_lead_conversion_without_user first.")
        return
    
    project = Project.objects.filter(client_email=lead.email, client__isnull=True).first()
    if not project:
        print("✗ Test project not found. Run test_lead_conversion_without_user first.")
        return
    
    print(f"✓ Found existing project: {project.title}")
    print(f"  - Client: {project.client} (should be None)")
    print(f"  - Client Email: {project.client_email}")
    
    # Simulate client registration
    user = User.objects.create_user(
        username='testclient',
        email=lead.email,
        password='testpass123',
        first_name=lead.first_name,
        last_name=lead.last_name
    )
    
    profile = UserProfile.objects.create(
        user=user,
        role='client'
    )
    
    print(f"✓ Created client user: {user.username}")
    
    # Link existing projects (this is what happens in client_register view)
    existing_projects = Project.objects.filter(
        client_email__iexact=user.email,
        client__isnull=True
    )
    
    print(f"✓ Found {existing_projects.count()} existing project(s) to link")
    
    for proj in existing_projects:
        proj.client = user
        proj.save()
        print(f"  - Linked project: {proj.title}")
    
    # Verify project is now linked
    project.refresh_from_db()
    print(f"✓ Project now linked to user: {project.client.username if project.client else 'None'}")
    
    return user

def test_staff_registration_approval():
    """Test staff self-registration with approval system"""
    print("\n=== Testing Staff Registration with Approval ===")
    
    # Create a staff user (simulating self-registration)
    staff_user = User.objects.create_user(
        username='teststaff',
        email='teststaff@example.com',
        password='testpass123',
        first_name='Test',
        last_name='Staff'
    )
    
    profile = UserProfile.objects.create(
        user=staff_user,
        role='crm',
        is_approved=False,  # Pending approval
        approval_requested_at=timezone.now()
    )
    
    print(f"✓ Created staff user: {staff_user.username}")
    print(f"  - Role: {profile.get_role_display()}")
    print(f"  - Approved: {profile.is_approved} (should be False)")
    print(f"  - Approval Requested: {profile.approval_requested_at}")
    
    # Check pending approvals count
    pending_count = User.objects.filter(
        is_superuser=False,
        profile__is_approved=False,
        profile__role__in=['crm', 'lead_manager', 'project_manager', 'employee']
    ).count()
    
    print(f"✓ Pending staff approvals: {pending_count}")
    
    # Simulate admin approval
    admin = User.objects.filter(is_superuser=True).first()
    if admin:
        profile.is_approved = True
        profile.approved_by = admin
        profile.approved_at = timezone.now()
        profile.save()
        
        print(f"✓ Staff approved by: {admin.username}")
        print(f"  - Approved: {profile.is_approved} (should be True)")
        print(f"  - Approved At: {profile.approved_at}")
    else:
        print("✗ No admin user found for approval simulation")
    
    return staff_user

def cleanup_test_data():
    """Clean up test data"""
    print("\n=== Cleaning Up Test Data ===")
    
    # Delete test users
    User.objects.filter(username__in=['testclient', 'teststaff']).delete()
    print("✓ Deleted test users")
    
    # Delete test projects
    Project.objects.filter(client_email='testlead@example.com').delete()
    print("✓ Deleted test projects")
    
    # Delete test leads
    Lead.objects.filter(email='testlead@example.com').delete()
    print("✓ Deleted test leads")
    
    # Delete test services
    Service.objects.filter(name='Test Service').delete()
    print("✓ Deleted test services")

def main():
    print("=" * 60)
    print("Lead Conversion & Staff Registration System Test")
    print("=" * 60)
    
    try:
        # Test 1: Lead conversion without user creation
        lead, project = test_lead_conversion_without_user()
        
        # Test 2: Client registration with project linking
        client_user = test_client_registration_with_project_linking()
        
        # Test 3: Staff registration with approval
        staff_user = test_staff_registration_approval()
        
        print("\n" + "=" * 60)
        print("✓ All tests completed successfully!")
        print("=" * 60)
        
        # Cleanup
        cleanup_input = input("\nClean up test data? (y/n): ")
        if cleanup_input.lower() == 'y':
            cleanup_test_data()
            print("\n✓ Test data cleaned up successfully!")
        
    except Exception as e:
        print(f"\n✗ Error during testing: {str(e)}")
        import traceback
        traceback.print_exc()

if __name__ == '__main__':
    main()
