"""
Test script for real-time notification system
"""
import os
import django

os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'myproject.settings')
django.setup()

from django.contrib.auth.models import User
from crm.models import Notification

def test_notifications():
    """Test notification system"""
    print("\n" + "="*60)
    print("NOTIFICATION SYSTEM TEST")
    print("="*60)
    
    # Get admin user
    admin = User.objects.filter(is_superuser=True).first()
    
    if not admin:
        print("\n⚠ No admin user found!")
        return
    
    print(f"\n✓ Testing notifications for: {admin.get_full_name()}")
    
    # Get all notifications
    all_notifications = Notification.objects.filter(user=admin)
    unread_notifications = all_notifications.filter(is_read=False)
    
    print(f"\nNotification Statistics:")
    print(f"  - Total notifications: {all_notifications.count()}")
    print(f"  - Unread notifications: {unread_notifications.count()}")
    print(f"  - Read notifications: {all_notifications.filter(is_read=True).count()}")
    
    # Show recent unread notifications
    if unread_notifications.exists():
        print(f"\n{'='*60}")
        print("UNREAD NOTIFICATIONS:")
        print(f"{'='*60}")
        
        for notif in unread_notifications[:5]:
            print(f"\n{notif.get_type_display()}:")
            print(f"  Title: {notif.title}")
            print(f"  Message: {notif.message}")
            print(f"  Created: {notif.created_at.strftime('%Y-%m-%d %H:%M:%S')}")
            print(f"  Link: {notif.link}")
    else:
        print("\n✓ No unread notifications")
    
    # Create a test notification
    print(f"\n{'='*60}")
    print("CREATING TEST NOTIFICATION")
    print(f"{'='*60}")
    
    test_notif = Notification.objects.create(
        user=admin,
        title='Test Notification',
        message='This is a test notification for the real-time system',
        type='new_message',
        link='/crm-admin/dashboard/'
    )
    
    print(f"\n✓ Created test notification: {test_notif.notification_id}")
    print(f"  Title: {test_notif.title}")
    print(f"  Message: {test_notif.message}")
    print(f"  Type: {test_notif.get_type_display()}")
    
    # Count after creation
    new_unread_count = Notification.objects.filter(user=admin, is_read=False).count()
    print(f"\n✓ New unread count: {new_unread_count}")
    
    print("\n" + "="*60)
    print("TEST COMPLETE")
    print("="*60)
    print("\nNext steps:")
    print("1. Visit http://127.0.0.1:8000/crm-admin/dashboard/")
    print("2. Check the notification bell icon (should show badge)")
    print("3. Click the bell to see the dropdown")
    print("4. Click on a notification to mark it as read")
    print("5. Click 'Mark all as read' to clear all notifications")
    print("\nFeatures:")
    print("- Auto-refresh every 30 seconds")
    print("- Sound notification for new notifications")
    print("- Toast popup for new notifications")
    print("- Click notification to navigate to link")
    print("- Badge shows unread count")

if __name__ == '__main__':
    test_notifications()
