"""
Test script for client ID verification system
"""
import os
import django

os.environ.setdefault('DJANGO_SETTINGS_MODULE', 'myproject.settings')
django.setup()

from django.contrib.auth.models import User
from crm.models import UserProfile

def test_verification_system():
    """Test the verification system"""
    print("=" * 60)
    print("CLIENT ID VERIFICATION SYSTEM TEST")
    print("=" * 60)
    
    # Check if admin exists
    admin = User.objects.filter(is_superuser=True).first()
    if not admin:
        print("❌ No admin user found!")
        return
    
    print(f"✓ Admin user: {admin.get_full_name()} ({admin.username})")
    
    # Check for clients
    clients = User.objects.filter(profile__role='client')
    print(f"\n✓ Total clients: {clients.count()}")
    
    # Check verification statuses
    pending = UserProfile.objects.filter(role='client', verification_status='pending').count()
    verified = UserProfile.objects.filter(role='client', verification_status='verified').count()
    rejected = UserProfile.objects.filter(role='client', verification_status='rejected').count()
    unverified = UserProfile.objects.filter(role='client', verification_status='unverified').count()
    
    print(f"\nVerification Status Breakdown:")
    print(f"  - Pending: {pending}")
    print(f"  - Verified: {verified}")
    print(f"  - Rejected: {rejected}")
    print(f"  - Unverified: {unverified}")
    
    # Show clients with ID proofs
    clients_with_id = UserProfile.objects.filter(role='client').exclude(id_proof='')
    print(f"\n✓ Clients with ID proof uploaded: {clients_with_id.count()}")
    
    if clients_with_id.exists():
        print("\nClients with ID Proof:")
        for profile in clients_with_id:
            print(f"  - {profile.user.get_full_name()}")
            print(f"    Status: {profile.get_verification_status_display()}")
            print(f"    Uploaded: {profile.id_proof_uploaded_at}")
            if profile.verified_by:
                print(f"    Verified by: {profile.verified_by.get_full_name()}")
                print(f"    Verified at: {profile.verified_at}")
            if profile.verification_notes:
                print(f"    Notes: {profile.verification_notes}")
            print()
    
    # Test URLs
    print("\nVerification URLs:")
    print(f"  - Admin Dashboard: http://127.0.0.1:8000/crm-admin/dashboard/")
    print(f"  - Admin Users: http://127.0.0.1:8000/crm-admin/users/")
    print(f"  - Client Profile: http://127.0.0.1:8000/client/profile/")
    
    print("\n" + "=" * 60)
    print("TEST COMPLETE")
    print("=" * 60)
    
    print("\nNext Steps:")
    print("1. Login as a client at http://127.0.0.1:8000/client/login/")
    print("2. Go to profile and upload an ID proof")
    print("3. Login as admin at http://127.0.0.1:8000/staff/login/")
    print("4. Check dashboard for pending verifications")
    print("5. Click 'Review' to approve or reject the verification")

if __name__ == '__main__':
    test_verification_system()
